from django.conf import settings
from django.forms import Form, IntegerField
from rest_framework.views import APIView
from rest_framework.status import HTTP_201_CREATED
from rest_framework.response import Response
from alto_django_utils.params import required_body_params, optional_body_params
from alto_django_utils.viewsets import CustomPagination
from alto_django_utils.permissions import IsClientUser, IsAuthenticated
from alto_django_utils.serializers import parse_field_value
from alto_django_kredit.models import Transaction
from alto_django_kredit.services import CreateTransaction as CreateTransactionService
from alto_django_kredit.serializers import TransactionSerializer

class GetTransactions(APIView):
  permission_classes = [IsClientUser]

  def get(self, request):
    transactions = request.user.client.transactions \
      .filter(kind__in = ['KC', 'POI']).order_by('-transacted_at')
    paginator = CustomPagination()
    page      = paginator.paginate_queryset(transactions, request)
    sideloads = {'bill_header': {'items': {}}}

    if 'alto_django_food' in settings.INSTALLED_APPS:
      sideloads['bill_header']['items']['product'] = {}

    serializer = TransactionSerializer(
      page, many = True, envelope = True, request_fields = sideloads
    )

    return paginator.get_paginated_response(serializer.data)

class GetTransactionKinds(APIView):
  permission_classes = [IsAuthenticated]

  def get(self, request):
    kinds = Transaction.objects.values_list('kind', flat = True).distinct()

    return Response({'kinds': list(kinds)})

class CreateTransaction(APIView):
  def post(self, request):
    client, kind, amount       = required_body_params(request, 'client', 'kind', 'amount')
    transacted_at, description = optional_body_params(request, 'transacted_at', 'description')
    amount                     = parse_field_value(TransactionSerializer, 'amount', amount)

    serializer = CreateTransactionService(
      client = client, kind = kind, amount = amount,
      transacted_at = transacted_at, description = description,
    )()

    return Response({ 'transaction': serializer.data }, status = HTTP_201_CREATED)
