from rest_framework.views import APIView
from rest_framework.fields import DateTimeField, DecimalField, IntegerField
from rest_framework.response import Response
from alto_django_utils.params import required_query_params, optional_query_params
from alto_django_utils.serializers import validate
from alto_django_kredit.services import GetClientsWithTransactionSums

class ClientsWithTransactionSums(APIView):
  def get(self, request):
    kind, start, end = self._get_required_params(request)
    min_sum, max_sum = self._get_sum_bound_params(request)
    page, page_size  = self._get_pagination_params(request)

    service = GetClientsWithTransactionSums(kind = kind, start = start, end = end,
                                            min_sum = min_sum, max_sum = max_sum)

    results, count = service(page = page, page_size = page_size)

    return Response({'count': count, 'results': list(results)})

  def _get_required_params(self, request):
    kind, start, end = required_query_params(request, 'kind', 'start', 'end')

    validate('start', start, DateTimeField())
    validate('end',   end,   DateTimeField())

    return kind, start, end

  def _get_sum_bound_params(self, request):
    min_sum, max_sum = optional_query_params(request, 'min_sum', 'max_sum')

    if min_sum: validate('min_sum', min_sum, DecimalField(max_digits = 12, decimal_places = 2))
    if max_sum: validate('max_sum', max_sum, DecimalField(max_digits = 12, decimal_places = 2))

    return min_sum, max_sum

  def _get_pagination_params(self, request):
    page, page_size  = optional_query_params(request, 'page', 'page_size')

    if page:      validate('page',      page,      IntegerField())
    if page_size: validate('page_size', page_size, IntegerField())

    return page, page_size
