from django.shortcuts import render, get_object_or_404
from django.utils.translation import gettext as _
from rest_framework.views import APIView
from rest_framework.fields import DateTimeField
from rest_framework.parsers import FormParser
from rest_framework.response import Response
from rest_framework.permissions import IsAuthenticated, AllowAny
from alto_django_utils.params import optional_query_param, required_query_param
from alto_django_utils.serializers import validate
from alto_django_kredit.forms import ShowBalanceForm
from alto_django_kredit.models import Client, Transaction
from alto_django_kredit.settings import settings

class GetBalances(APIView):
  permission_classes = [IsAuthenticated]

  def get(self, request):
    if request.user.client:
      client = request.user.client
    else:
      client_id = required_query_param(request, 'client_id')
      client    = get_object_or_404(Client, id = client_id)

    at = optional_query_param(request, 'at')
    if at: validate('at', at, DateTimeField())

    kinds     = client.transactions.all().values_list('kind', flat = True).distinct()
    balances  = { kind: str(Transaction.balance_for(client, kind, at = at)) for kind in kinds }

    return Response({ 'balances': balances })

class ShowBalance(APIView):
  permission_classes = [AllowAny]
  parser_classes     = [FormParser]

  def get(self, request):
    return render(request, 'kredit/show_balance_form.html',
                  { 'form': ShowBalanceForm() })

  def post(self, request):
    form          = ShowBalanceForm(request.data)
    client_finder = settings.client_finder

    if form.is_valid():
      try:
        client = client_finder.client_for_card_and_email(form.cleaned_data['card_number'],
                                                         form.cleaned_data['email'])
      except client_finder.ClientNotFound:
        form.add_error(None, 'Klient se zadanými údaji nebyl nalezen.')

        return render(request, 'kredit/show_balance_form.html', { 'form': form })
      except client_finder.MultipleClientsFound:
        form.add_error(None, 'Bylo nalezeno několik klientů se zadanými údaji. '
                       'Kontaktujte prosím klientské centrum.')

        return render(request, 'kredit/show_balance_form.html', { 'form': form })

      balance = Transaction.balance_for(client, 'POI')

      return render(request, 'kredit/show_balance.html', { 'balance': balance })
    else:
      return render(request, 'kredit/show_balance_form.html', { 'form': form })
