from django.urls import reverse
from django.contrib.auth import get_user_model
from rest_framework.test import APITestCase
from alto_django_kredit.models import Client, Transaction
from alto_django_client_auth.tests.helpers import ClientAuthHelpers, auth_header

User = get_user_model()

def refresh(entity):
  entity.refresh_from_db()

  return entity

class HistoricTransactionsTests(APITestCase, ClientAuthHelpers):
  def setUp(self):
    User.objects.create_user('admin', 'admin@test.cz', 'admin', is_superuser = True)

    self.token   = self.get_token(username = 'admin', password = 'admin').data['token']
    self.client1 = Client.objects.create(client_id = '123', card_id = '123')
    self.client2 = Client.objects.create(client_id = '234', card_id = '234')

  def test_create_with_new_transaction(self):
    Transaction.objects.create(client = self.client1, kind = 'KC', amount = 100, balance = 100)

    response = self.client.post(reverse('alto_django_kredit:historic_transactions-list'),
                                { 'client': self.client1.id,
                                  'kind': 'KC',
                                  'amount': 100 },
                                **auth_header(self.token))

    assert response.status_code               == 201
    assert Transaction.objects.count()        == 2
    assert Transaction.objects.last().balance == 200

  def test_create_with_historic_transaction(self):
    tr1 = Transaction.objects.create(client = self.client1, kind = 'KC',
                                     amount = 100, balance = 100,
                                     transacted_at = '2010-10-10T12:00:00Z')
    tr2 = Transaction.objects.create(client = self.client1, kind = 'KC',
                                     amount = 100, balance = 200,
                                     transacted_at = '2012-12-12T12:00:00Z')

    response = self.client.post(reverse('alto_django_kredit:historic_transactions-list'),
                                { 'client': self.client1.id,
                                  'kind': 'KC',
                                  'amount': 100,
                                  'transacted_at': '2011-11-11T12:00:00Z' },
                                **auth_header(self.token))

    assert response.status_code        == 201
    assert Transaction.objects.count() == 3
    assert refresh(tr2).balance         == 300

  def test_update_changing_amount(self):
    tr1 = Transaction.objects.create(client = self.client1, kind = 'KC',
                                     amount = 100, balance = 100,
                                     transacted_at = '2010-10-10T12:00:00Z')
    tr2 = Transaction.objects.create(client = self.client1, kind = 'KC',
                                     amount = 100, balance = 200,
                                     transacted_at = '2011-11-11T12:00:00Z')
    tr3 = Transaction.objects.create(client = self.client1, kind = 'KC',
                                     amount = 100, balance = 300,
                                     transacted_at = '2012-12-12T12:00:00Z')

    response = self.client.patch(reverse('alto_django_kredit:historic_transactions-detail',
                                         args = [tr2.id]),
                                 { 'amount': 200 },
                                 **auth_header(self.token))

    assert response.status_code == 200
    assert refresh(tr2).amount  == 200
    assert refresh(tr2).balance == 300
    assert refresh(tr3).balance == 400

  def test_update_changing_transacted_at(self):
    tr1 = Transaction.objects.create(client = self.client1, kind = 'KC',
                                     amount = 100, balance = 100,
                                     transacted_at = '2010-10-10T12:00:00Z')
    tr2 = Transaction.objects.create(client = self.client1, kind = 'KC',
                                     amount = 100, balance = 200,
                                     transacted_at = '2011-11-11T12:00:00Z')
    tr3 = Transaction.objects.create(client = self.client1, kind = 'KC',
                                     amount = 100, balance = 300,
                                     transacted_at = '2012-12-12T12:00:00Z')

    response = self.client.patch(reverse('alto_django_kredit:historic_transactions-detail',
                                         args = [tr3.id]),
                                 { 'transacted_at': '2009-09-09T12:00:00Z' },
                                 **auth_header(self.token))

    assert response.status_code == 200
    assert refresh(tr1).balance == 200
    assert refresh(tr2).balance == 300
    assert refresh(tr3).balance == 100

  def test_update_changing_kind(self):
    kc1  = Transaction.objects.create(client = self.client1, kind = 'KC',
                                      amount = 100, balance = 100,
                                      transacted_at = '2010-10-10T12:00:00Z')
    kc2  = Transaction.objects.create(client = self.client1, kind = 'KC',
                                      amount = 100, balance = 200,
                                      transacted_at = '2011-11-11T12:00:00Z')
    poi1 = Transaction.objects.create(client = self.client1, kind = 'POI',
                                      amount = 10, balance = 10,
                                      transacted_at = '2012-12-12T12:00:00Z')
    poi2 = Transaction.objects.create(client = self.client1, kind = 'POI',
                                      amount = 10, balance = 20,
                                      transacted_at = '2012-12-13T12:00:00Z')

    response = self.client.patch(reverse('alto_django_kredit:historic_transactions-detail',
                                         args = [kc1.id]),
                                 { 'kind': 'POI' },
                                 **auth_header(self.token))

    assert response.status_code  == 200
    assert refresh(kc1).kind     == 'POI'
    assert refresh(kc2).balance  == 100
    assert refresh(poi1).balance == 110
    assert refresh(poi2).balance == 120

  def test_update_changing_client(self):
    cl11 = Transaction.objects.create(client = self.client1, kind = 'KC',
                                      amount = 100, balance = 100,
                                      transacted_at = '2010-10-10T12:00:00Z')
    cl12 = Transaction.objects.create(client = self.client1, kind = 'KC',
                                      amount = 100, balance = 200,
                                      transacted_at = '2011-11-11T12:00:00Z')
    cl21 = Transaction.objects.create(client = self.client2, kind = 'KC',
                                      amount = 10, balance = 10,
                                      transacted_at = '2012-12-12T12:00:00Z')
    cl22 = Transaction.objects.create(client = self.client2, kind = 'KC',
                                      amount = 10, balance = 20,
                                      transacted_at = '2012-12-13T12:00:00Z')

    response = self.client.patch(reverse('alto_django_kredit:historic_transactions-detail',
                                         args = [cl11.id]),
                                 { 'client': self.client2.id },
                                 **auth_header(self.token))

    assert response.status_code  == 200
    assert refresh(cl11).client  == self.client2
    assert refresh(cl12).balance == 100
    assert refresh(cl21).balance == 110
    assert refresh(cl22).balance == 120

  def test_delete(self):
    tr1 = Transaction.objects.create(client = self.client1, kind = 'KC',
                                     amount = 100, balance = 100,
                                     transacted_at = '2010-10-10T12:00:00Z')
    tr2 = Transaction.objects.create(client = self.client1, kind = 'KC',
                                     amount = 100, balance = 200,
                                     transacted_at = '2011-11-11T12:00:00Z')
    tr3 = Transaction.objects.create(client = self.client1, kind = 'KC',
                                     amount = 100, balance = 300,
                                     transacted_at = '2012-12-12T12:00:00Z')

    response = self.client.delete(reverse('alto_django_kredit:historic_transactions-detail',
                                          args = [tr1.id]),
                                  **auth_header(self.token))

    assert response.status_code        == 204
    assert Transaction.objects.count() == 2
    assert refresh(tr2).balance        == 100
    assert refresh(tr3).balance        == 200
