import pydash
from copy import deepcopy
from django.utils import timezone
from django.utils.translation import gettext as _
from rest_framework.exceptions import ValidationError
from rest_framework.serializers import DecimalField
from alto_django_utils.models import exclusive_access_to
from alto_django_utils.serializers import serializer_for
from .models import Client, ClientInfo, Transaction
from .queries import TransactionRecomputer

ClientSerializer          = serializer_for(Client)
ClientInfoSerializer      = serializer_for(ClientInfo)
BaseTransactionSerializer = serializer_for(Transaction)

class TransactionSerializer(BaseTransactionSerializer):
  amount  = DecimalField(decimal_places = 2, max_digits = 12, required = False)
  balance = DecimalField(decimal_places = 2, max_digits = 12, required = False)

  def validate(self, data):
    self._validate_transacted_at(data)

    if 'amount' in data and 'balance' in data:
      if not (data['balance'] == self.balance_for(data) + data['amount']):
        raise ValidationError({'amount':  [_("Doesn't match 'balance' - current balance.")],
                               'balance': [_("Doesn't match current balance + 'amount'.")]})

    if 'amount' not in data and 'balance' not in data and not self.partial:
      message = _("At least one of 'amount' and 'balance' must be provided.")
      raise ValidationError({ 'amount': [message], 'balance': [message] })

    return data

  def _validate_transacted_at(self, data):
    if 'transacted_at' in data:
      last_transaction = Transaction.last_for(client = data['client'], kind = data['kind'])

      if last_transaction and data['transacted_at'] < last_transaction.transacted_at:
        message = _('Newer transaction exists, this would create inconsistent transaction history.')
        raise ValidationError({ 'transacted_at': [message] })

  def save(self, **kwargs):
    data = self.validated_data

    with exclusive_access_to(Transaction):
      if 'amount' in data and 'balance' not in data:
        self.validated_data['balance'] = self.balance_for(data) + data['amount']

      if 'balance' in data and 'amount' not in data:
        self.validated_data['amount'] = data['balance'] - self.balance_for(data)

      result = super().save(**kwargs)

    return result

  def balance_for(self, data):
    return Transaction.balance_for(data['client'], data['kind'])

class HistoricTransactionSerializer(TransactionSerializer):
  @classmethod
  def get_name(self):
    return 'historic_transaction'

  @classmethod
  def get_plural_name(self):
    return 'historic_transactions'

  def get_resource_key(cls):
    return 'historic_transactions'

  def create(self, validated_data):
    client, kind, transacted_at = pydash.at(validated_data, 'client', 'kind', 'transacted_at')

    with TransactionRecomputer(client, kind) as recomputer:
      last_transaction = Transaction.last_for(client = client, kind = kind)

      instance = super().create(validated_data)

      if not last_transaction or transacted_at > last_transaction.transacted_at:
        recomputer.needs_recompute = False

    return instance

  def update(self, instance, validated_data):
    old_instance = deepcopy(instance)

    with TransactionRecomputer(validated_data.get('client') or instance.client,
                               validated_data.get('kind')   or instance.kind) as recomputer:
      instance = super().update(instance, validated_data)

      if instance.client != old_instance.client or instance.kind != old_instance.kind:
        recomputer.recompute(old_instance.client, old_instance.kind)

      if (instance.client        == old_instance.client and
          instance.kind          == old_instance.kind   and
          instance.amount        == old_instance.amount and
          instance.transacted_at == old_instance.transacted_at):
        recomputer.needs_recompute = False

    return instance

  def balance_for(self, data):
    return Transaction.balance_for_at(data.get('client')        or self.instance.client,
                                      data.get('kind')          or self.instance.kind,
                                      data.get('transacted_at') or self.instance.transacted_at)

  def _validate_transacted_at(self, data):
    if not self.partial and 'transacted_at' not in data:
      data['transacted_at'] = timezone.now()
