import re
from datetime import datetime, timedelta
from pytest   import mark

from alto_eet_client.results import (ConnectionFailed, ParsingFailed, SigningFailed,
                                     CommunicationError, RequestAccepted, RequestRejected)
from alto_eet_client.tests.common import FIK_REGEX, BKP_REGEX, PKP_REGEX, UUID_REGEX

class TestReportTransaction:
  def test_success(self, api_client, transaction):
    result = api_client.report_transaction(transaction)

    assert result.is_success
    assert isinstance(result, RequestAccepted)
    assert result.test            is True
    # assert result.warning         is None
    assert result.transaction.dic == transaction.dic
    assert re.fullmatch(FIK_REGEX,  result.fik)
    assert re.fullmatch(BKP_REGEX,  result.bkp)
    assert re.fullmatch(PKP_REGEX,  result.pkp)
    assert re.fullmatch(UUID_REGEX, result.uuid)

  def test_unreachable_api(self, unreachable_api_client, transaction):
    result = unreachable_api_client.report_transaction(transaction)

    assert result.is_failure
    assert isinstance(result, ConnectionFailed)
    assert ('connection refused'    in result.detail.lower() or
            'name does not resolve' in result.detail.lower())

  def test_unparsable_transaction(self, api_client, invalid_transaction):
    result = api_client.report_transaction(invalid_transaction)

    assert result.is_failure
    assert isinstance(result, ParsingFailed)
    assert re.match('bad dic.* attribute format', result.detail)

  @mark.skip('requires a second EET API running on localhost:8001 with incorrect `eet_server` '
             'setting, or some other obstacle when connecting to fiscal server')
  def test_fiscal_server_communication_error(self, misconfigured_api_client, transaction):
    result = misconfigured_api_client.report_transaction(transaction)

    assert result.is_failure
    assert isinstance(result, CommunicationError)
    assert result.transaction.dic == transaction.dic
    assert re.fullmatch(BKP_REGEX,  result.bkp)
    assert re.fullmatch(PKP_REGEX,  result.pkp)

  # TODO: test RequestRejected result type
  # - request format is checked by our API, so these kinds of errors don't hit the fiscal server
  # - other then format, the fiscal server (at least the testing one) doesn't do many checks,
  #   so we can't really test this, unless we find some special case that the server actually checks
