import time
import requests, requests_mock
from alto_gp_webpay.config  import Config
from alto_gp_webpay.gateway import Gateway
from alto_gp_webpay.test.pages.payment import PaymentPage

class TestGetPaymentStatus:
  config  = Config.load()
  gateway = Gateway(config)

  def test_successful_payment(self):
    init_result  = self.gateway.create_payment(amount = 100)

    result = self.gateway.get_payment_status(init_result.payment_number)

    assert result.is_success
    assert result.status     == 'PENDING_AUTHORIZATION'
    assert result.sub_status == 'INITIATED'

    PaymentPage.pay(init_result.redirect)

    result = self.gateway.get_payment_status(init_result.payment_number)

    assert result.is_success
    assert result.status     == 'CAPTURED'
    assert result.sub_status == 'PENDING_CAPTURE_SETTLEMENT'

  def test_failed_payment(self):
    init_result  = self.gateway.create_payment(amount = 100)

    result = self.gateway.get_payment_status(init_result.payment_number)

    assert result.is_success
    assert result.status     == 'PENDING_AUTHORIZATION'
    assert result.sub_status == 'INITIATED'

    PaymentPage.pay(init_result.redirect, fail = True)

    result = self.gateway.get_payment_status(init_result.payment_number)

    assert result.is_success
    assert result.status     == 'UNPAID'
    assert result.sub_status == 'DECLINED'

  def test_connection_failure(self):
    init_result = self.gateway.create_payment(amount = 100)

    with requests_mock.Mocker() as mocker:
      mocker.register_uri(requests_mock.ANY, requests_mock.ANY,
                          exc = requests.exceptions.ConnectionError)

      result = self.gateway.get_payment_status(init_result.payment_number)

    assert not result.is_success
    assert result.message == 'Technical problem: Service connection failed'

  def test_bad_payment_number_format(self):
    result = self.gateway.get_payment_status('asdf')

    assert not result.is_success
    assert result.message == 'Incorrect message format'

  def test_nonexistent_payment_number(self):
    result = self.gateway.get_payment_status(999999999999999)

    assert not result.is_success
    assert result.message == 'Object not found: ORDERNUMBER'

  def test_bad_merchant_number(self):
    config  = Config.load()
    config.merchant_number = 12345
    gateway = Gateway(config)

    result = gateway.get_payment_status(1)

    assert not result.is_success
    assert result.message == 'Unknown merchant'

  # TODO: add logging
