import re

class Result:
  PRIMARY_CODES = {
    0:    ["OK", "OK"],
    1:    ["Field too long",                       "Pole příliš dlouhé"],
    2:    ["Field too short",                      "Pole příliš krátké"],
    3:    ["Incorrect content of field",           "Chybný obsah pole"],
    4:    ["Field is null",                        "Pole je prázdné"],
    5:    ["Missing required field",               "Chybí povinné pole"],
    6:    ["Field doesn't exist",                  "Pole neexistuje"],
    7:    ["Incorrect message format",             "Chybný formát zprávy"],
    11:   ["Unknown merchant",                     "Neznámý obchodník"],
    14:   ["Duplicate order number",               "Duplikátní číslo objednávky"],
    15:   ["Object not found",                     "Objekt nenalezen"],
    17:   ["Amount to deposit exceeds approved amount",
           "Částka k úhradě překročila autorizovanou částku"],
    18:   ["Total sum of credited amounts exceeded deposited amount",
           "Součet kreditovaných částek překročil uhrazenou částku"],
    20:   ["Object not in valid state for operation",
           "Objekt není ve stavu odpovídajícím této operaci"],
    25:   ["Operation not allowed for user",       "Uživatel není oprávněn k provedení operace"],
    26:   ["Technical problem in connection to authorization centre",
           "Technický problém při spojení s autorizačním centrem"],
    27:   ["Incorrect order type",                 "Chybný typ objednávky"],
    28:   ["Declined in 3D",                       "Zamítnuto v 3D"],
    30:   ["Declined in AC",                       "Zamítnuto v autorizačním centru"],
    31:   ["Wrong digest",                         "Chybný podpis"],
    35:   ["Session expired",                      "Vypršel časový limit"],
    50:   ["The cardholder cancelled the payment", "Držitel karty zrušil platbu"],
    200:  ["Additional info request",              "Žádost o doplňující informace"],
    1000: ["Technical problem",                    "Technický problém"]
  }

  SECONDARY_CODES = {
    0:    ["No meaning",                                 "Bez významu"],
    1:    ["ORDERNUMBER",                                "Číslo objednávky"],
    2:    ["MERCHANTNUMBER",                             "Kód obchodníka"],
    6:    ["AMOUNT",                                     "Částka"],
    7:    ["CURRENCY",                                   "Měna"],
    8:    ["DEPOSITFLAG",                                "Zaplatit okamžitě"],
    10:   ["MERORDERNUM",                                "Číslo objednávky u obchodníka"],
    11:   ["CREDITNUMBER",                               "Kreditní číslo"],
    12:   ["OPERATION",                                  "Operace"],
    18:   ["BATCH",                                      "Dávka"],
    22:   ["ORDER",                                      "Objednávka"],
    24:   ["URL",                                        "URL"],
    25:   ["MD",                                         "Dodatečná data"],
    26:   ["DESC",                                       "Popis"],
    34:   ["DIGEST",                                     "Podpis"],
    1001: ["Card blocked",                               "Karta blokována"],
    1002: ["Declined",                                   "Autorizace zamítnuta"],
    1003: ["Card problem",                               "Problém karty"],
    1004: ["Technical problem in authorization process", "Technický problém při autorizaci"],
    1005: ["Account problem",                            "Problém účtu"],
    3000: ["Cardholder not authenticated in 3D.",        "Vydavatel karty není zapojen do 3D."],
    3001: ["Authenticated",                              "Držitel karty ověřen."],
    3002: ["Issuer or Cardholder not participating in 3D.",
           "Vydavatel karty nebo karta není zapojena do 3D."],
    3004: ["Issuer not participating or Cardholder not enrolled.",
           "Vydavatel karty není zapojen do 3D nebo karta nebyla aktivována."],
    3005: ["Technical problem during Cardholder authentication.",
           "Technický problém při ověření držitele karty."],
    3006: ["Technical problem during Cardholder authentication.",
           "Technický problém při ověření držitele karty."],
    3007: ["Acquirer technical problem. Contact the merchant.",
           "Technický problém v systému zúčtující banky. Kontaktujte obchodníka."],
    3008: ["Unsupported card product.", "Použit nepodporovaný karetní produkt."],
    4000: ["Service connection failed", "Připojení ke službě selhalo"]
  }

  def __init__(self, primary_code = 0, secondary_code = 0, payment_number = None):
    self.primary_code   = primary_code
    self.secondary_code = secondary_code
    self.payment_number = payment_number

  @property
  def is_success(self):
    return self.primary_code == 0

  def get_message(self, translated = False):
    tr_index        = 1 if translated else 0
    primary_message = self.PRIMARY_CODES[self.primary_code][tr_index]

    if self.secondary_code == 0:
      return primary_message
    else:
      secondary_message = self.SECONDARY_CODES[self.secondary_code][tr_index]

      return f"{primary_message}: {secondary_message}"

  message = property(get_message)

  @property
  def translated_message(self):
    return self.get_message(translated = True)

  def __str__(self):
    return f"<{self.__class__.__name__} '{self.message}' {self.__dict__}>"

  __repr__ = __str__

class EchoResult(Result): pass

class CreatePaymentResult(Result):
  def __init__(self, primary_code = 0, secondary_code = 0,
               *, payment_number, redirect = None):
    super().__init__(primary_code, secondary_code, payment_number)

    self.redirect = redirect

  @classmethod
  def from_response(cls, payment_number, response):
    if response.status_code == 302:
      return cls(0, 0,
                 payment_number = payment_number,
                 redirect     = response.headers['Location'])

    else:
      match = re.search('PRCODE=(\d+),\s*SRCODE=(\d+)', response.text)
      primary, secondary = list(map(int, match.group(1, 2)))

      return cls(payment_number = payment_number, primary_code = primary, secondary_code = secondary)

class GetPaymentStatusResult(Result):
  def __init__(self, primary_code = 0, secondary_code = 0,
               *, payment_number, message_id, status = None, sub_status = None):
    super().__init__(primary_code, secondary_code, payment_number)

    self.message_id = message_id
    self.status     = status
    self.sub_status = sub_status

  @classmethod
  def from_response(cls, payment_number, response):
    return cls(0, 0, payment_number = payment_number, message_id = response.messageId,
               status = response.status, sub_status = response.subStatus)

  @classmethod
  def from_service_exception(cls, payment_number, exception):
    type_prefix    = '{http://gpe.cz/pay/pay-ws/proc/v1/type}'
    message_id     = exception.find(type_prefix + 'messageId').text
    primary_code   = int(exception.find(type_prefix + 'primaryReturnCode').text)
    secondary_code = int(exception.find(type_prefix + 'secondaryReturnCode').text)

    return cls(primary_code, secondary_code,
               payment_number = payment_number, message_id = message_id)

class ProcessResult(Result):
  @classmethod
  def from_query(cls, params):
    return cls(int(params['PRCODE']), int(params['SRCODE']), params['ORDERNUMBER'])
