import time, base64, requests, zeep
from decimal import Decimal
from .key    import Key
from .logger import Logger
from .result import EchoResult, CreatePaymentResult, GetPaymentStatusResult, ProcessResult

class Gateway:
  PROD_GW_URL  = 'https://3dsecure.gpwebpay.com/pgw/order.do'
  TEST_GW_URL  = 'https://test.3dsecure.gpwebpay.com/pgw/order.do'
  PROD_WS_URL  = 'https://3dsecure.gpwebpay.com/pay-ws/v1/PaymentService'
  TEST_WS_URL  = 'https://test.3dsecure.gpwebpay.com/pay-ws/v1/PaymentService'
  SERVICE_NAME = '{http://gpe.cz/pay/pay-ws/proc/v1}PaymentServiceSoap'

  def __init__(self, config):
    self.config  = config
    self.logger  = Logger(config.log_dir)
    self.key     = Key(config.key_path, config.key_password)
    self.gw_url  = self.TEST_GW_URL if self.config.test else self.PROD_GW_URL
    self.ws_url  = self.TEST_WS_URL if self.config.test else self.PROD_WS_URL
    self.client  = zeep.Client(wsdl = self.config.schema_path)
    self.service = self.client.create_service(self.SERVICE_NAME, self.ws_url)
    self.types   = self.client.type_factory('ns0')

  @property
  def base_data(self):
    return {'MERCHANTNUMBER': self.config.merchant_number,
            'URL':            self.config.return_url}

  def echo(self):
    try:
      self.service.echo()

      return EchoResult()
    except requests.exceptions.ConnectionError as error:
      return EchoResult(1000, 4000)

  def create_payment(self, payment_number = None, *, amount, deposit_flag = 1):
    payment_number = payment_number or int(time.time() * 1000)

    data = {'OPERATION':      'CREATE_ORDER',
            'ORDERNUMBER':    payment_number,
            'AMOUNT':         Decimal(amount) * 100,
            'DEPOSITFLAG':    deposit_flag}

    data.update(self.base_data)
    self._add_digest(data)

    self.logger.log_create_payment_request(self.gw_url, data)
    response = requests.post(self.gw_url, data = data, allow_redirects = False)
    self.logger.log_create_payment_response(response)
    result   = CreatePaymentResult.from_response(payment_number, response)
    self.logger.log_result(result)

    return result

  def process(self, query_params):
    self.logger.log_process(query_params)
    result = ProcessResult.from_query(query_params)
    self.logger.log_result(result)

    return result

  def get_payment_status(self, payment_number):
    message_id = int(time.time() * 1000000)
    signature  = self._compute_digest([message_id, self.config.provider,
                                      self.config.merchant_number, payment_number])

    request = self.types.PaymentStatusRequest(messageId      = message_id,
                                              provider       = self.config.provider,
                                              merchantNumber = self.config.merchant_number,
                                              paymentNumber  = payment_number,
                                              signature      = signature)

    try:
      self.logger.log_get_payment_status_request(request)
      response = self.service.getPaymentStatus(request)
      self.logger.log_get_payment_status_response(response)

      result = GetPaymentStatusResult.from_response(payment_number, response)
    except zeep.exceptions.Fault as error:
      self.logger.log_get_payment_status_error(error)

      result = GetPaymentStatusResult.from_service_exception(payment_number, error.detail[0])
    except requests.exceptions.ConnectionError as error:
      self.logger.log_get_payment_status_error(error)

      result = GetPaymentStatusResult(1000, 4000,
                                    payment_number = payment_number, message_id = message_id)

    self.logger.log_result(result)

    return result

  def _add_digest(self, data):
    parts = [data['MERCHANTNUMBER'],
             data['OPERATION'],
             data['ORDERNUMBER'],
             data['AMOUNT'],
             data['DEPOSITFLAG'],
             data['URL']]

    data['DIGEST'] = base64.b64encode(self._compute_digest(parts))

  def _compute_digest(self, parts):
    to_sign = bytes('|'.join(map(str, parts)), 'ascii')

    return self.key.sign(to_sign)
