from numbers import Number
# ParsingError isn't used here, but is passed to caller, so it makes sense to import from here
from alto_utils.params import Params, ParsingError
from alto_eet_common.models import Transaction

DIC_REGEX      = r'^[A-Z]{2}\d{8,10}$'
CANTEEN_REGEX  = r'^[1-9][0-9]{0,5}$'
REGISTER_REGEX = r'^[0-9a-zA-Z\.,:;/#\-_ ]{1,20}$'
BILL_NUM_REGEX = r'^[0-9a-zA-Z\.,:;/#\-_ ]{1,25}$'

def format_money(value):
  if   value is None:             return value
  elif isinstance(value, str):    return value
  elif isinstance(value, Number): return '{0:.2f}'.format(value)
  else: raise Exception('unexpected money type: {val_type}'.format(val_type = type(value)))

class TransactionSerializer:
  @staticmethod
  def from_params(params):
    params = Params(params)

    return Transaction(
      dic              = params.get('dic_popl' ).required.string.format(DIC_REGEX).value,
      entrustor_dic    = params.get('dic_poverujiciho'  ).string.format(DIC_REGEX).value,
      canteen_id       = params.get('id_provoz').required.string.format(CANTEEN_REGEX).value,
      register_id      = params.get('id_pokl'  ).required.string.format(REGISTER_REGEX).value,
      bill_number      = params.get('porad_cis').required.string.format(BILL_NUM_REGEX).value,

      transacted_at    = params.get('dat_trzby' ).required.timestamp.value,
      sent_at          = params.get('dat_odesl'          ).timestamp.value,

      first_attempt    = params.get('prvni_zaslani', True).boolean.value,
      verification     = params.get('overeni',      False).boolean.value,

      total            = params.get('celk_trzba').required.money.value,
      exempt_vat_base  = params.get('zakl_nepodl_dph').money.value,
      vat1_base        = params.get('zakl_dan1').money.value,
      vat2_base        = params.get('zakl_dan2').money.value,
      vat3_base        = params.get('zakl_dan3').money.value,
      vat1             = params.get('dan1').money.value,
      vat2             = params.get('dan2').money.value,
      vat3             = params.get('dan3').money.value,
      travel_services  = params.get('cest_sluz').money.value,
      used_goods_vat1  = params.get('pouzit_zboz1').money.value,
      used_goods_vat2  = params.get('pouzit_zboz2').money.value,
      used_goods_vat3  = params.get('pouzit_zboz3').money.value,
      deposit          = params.get('urceno_cerp_zuct').money.value,
      withdraw         = params.get('cerp_zuct').money.value)

  @staticmethod
  def to_params(transaction):
    result = { 'dic_popl'         : transaction.dic,
               'id_provoz'        : transaction.canteen_id,
               'id_pokl'          : transaction.register_id,
               'porad_cis'        : str(transaction.bill_number),
               'dat_trzby'        : transaction.transacted_at.isoformat(),
               'dat_odesl'        : transaction.sent_at.isoformat(),
               'prvni_zaslani'    : transaction.first_attempt,
               'overeni'          : transaction.verification,
               'celk_trzba'       : format_money(transaction.total),
               'zakl_nepodl_dph'  : format_money(transaction.exempt_vat_base),
               'zakl_dan1'        : format_money(transaction.vat1_base),
               'dan1'             : format_money(transaction.vat1),
               'zakl_dan2'        : format_money(transaction.vat2_base),
               'dan2'             : format_money(transaction.vat2),
               'zakl_dan3'        : format_money(transaction.vat3_base),
               'dan3'             : format_money(transaction.vat3),
               'cest_sluz'        : format_money(transaction.travel_services),
               'pouzit_zboz1'     : format_money(transaction.used_goods_vat1),
               'pouzit_zboz2'     : format_money(transaction.used_goods_vat2),
               'pouzit_zboz3'     : format_money(transaction.used_goods_vat3),
               'urceno_cerp_zuct' : format_money(transaction.deposit),
               'cerp_zuct'        : format_money(transaction.withdraw) }

    if transaction.entrustor_dic: result['dic_poverujiciho'] = transaction.entrustor_dic

    return result
