from alto_utils.http import HTTPResponse
from alto_eet_common.results import (ParsingFailed, CommunicationError, RequestAccepted,
                                     RequestRejected, SigningFailed)
from .transaction import TransactionSerializer

class ResultSerializer:
  SUCCESS_MESSAGE             = 'transaction successfully reported'
  PARSING_FAILED_MESSAGE      = 'failed to parse request'
  SIGNING_FAILED_MESSAGE      = 'failed to sign request'
  REQUEST_REJECTED_MESSAGE    = 'request rejected by fiscal server'
  # TODO: once EET server uses alto_eet_common replace this with a more descriptive message,
  # like "failed to connect to fiscal server"
  # - ask JQ whether he depends on the specific message string!
  COMMUNICATION_ERROR_MESSAGE = 'failed to report transaction'

  def __init__(self, result, transaction = None):
    self.result      = result
    self.transaction = transaction


  @classmethod
  def from_response(cls, response):
    data = response.data

    if response.status == 200 and data['message'] == cls.SUCCESS_MESSAGE:
      transaction = TransactionSerializer.from_params(data['transaction'])
      result      = RequestAccepted(transaction = transaction,
                                    uuid        = data['uuid'],
                                    pkp         = data['pkp'],
                                    bkp         = data['bkp'],
                                    fik         = data['fik'],
                                    warning     = data['warning'],
                                    test        = data['test'])

      return cls(result, transaction)

    if response.status == 400:
      if data['message'] == cls.PARSING_FAILED_MESSAGE:
        result = ParsingFailed(detail = data['detail'])

        return cls(result)

      if data['message'] == cls.REQUEST_REJECTED_MESSAGE:
        transaction = TransactionSerializer.from_params(data['transaction'])
        result      = RequestRejected(detail = data['detail'], transaction = transaction)

        return cls(result, transaction)

    if response.status == 424:
      if data['message'] == cls.SIGNING_FAILED_MESSAGE:
        transaction = TransactionSerializer.from_params(data['transaction'])
        result      = SigningFailed(detail = data['detail'], transaction = transaction)

        return cls(result, transaction)

      if data['message'] == cls.COMMUNICATION_ERROR_MESSAGE:
        transaction = TransactionSerializer.from_params(data['transaction'])
        result      = CommunicationError(detail = data['detail'], transaction = transaction,
                                         pkp = data['pkp'], bkp = data['bkp'])

        return cls(result, transaction)

    raise Exception('unexpected response')


  def to_response(self):
    body = self.get_body()

    if self.transaction:
      body.update({ 'transaction': TransactionSerializer.to_params(self.transaction) })

    return HTTPResponse(status = self.get_status(), data = body)

  def get_body(self):
    if self.result.is_success:
      return self._success(self.SUCCESS_MESSAGE)
    elif isinstance(self.result, ParsingFailed):
      return self._failure_without_codes(self.PARSING_FAILED_MESSAGE)
    elif isinstance(self.result, SigningFailed):
      return self._failure_without_codes(self.SIGNING_FAILED_MESSAGE)
    elif isinstance(self.result, RequestRejected):
      return self._failure_without_codes(self.REQUEST_REJECTED_MESSAGE)
    elif isinstance(self.result, CommunicationError):
      return self._failure_with_codes(self.COMMUNICATION_ERROR_MESSAGE)
    else:
      # should never happen
      raise ValueError('{result} cannot be converted to response body'
                       .format(result = repr(self.result)))

  def get_status(self):
    if self.result.is_success:
      return 200
    elif isinstance(self.result, (ParsingFailed, RequestRejected)):
      return 400
    elif isinstance(self.result, (SigningFailed, CommunicationError)):
      return 424
    else:
      # should never happen
      raise ValueError('{result} cannot be converted to response status'
                       .format(result = repr(self.result)))

  def _success(self, message):
    return { 'message' : message,
             'fik'     : self.result.fik,
             'uuid'    : self.result.uuid,
             'bkp'     : self.result.bkp,
             'pkp'     : self.result.pkp,
             'test'    : self.result.test,
             'warning' : self.result.warning }

  def _failure_with_codes(self, message):
    return { 'message' : message,
             'detail'  : self.result.detail,
             'bkp'     : self.result.bkp,
             'pkp'     : self.result.pkp }

  def _failure_without_codes(self, message):
    return { 'message' : message,
             'detail'  : self.result.detail }
