import pytest
from pydash import _
from django.conf import settings
from django.urls import reverse
from django.contrib.auth import get_user_model
from rest_framework.test import APITestCase
from alto_django_base_auth.tests.helpers import BaseAuthHelpers, auth_header

if 'alto_django_food' in settings.INSTALLED_APPS:
  from alto_django_food.models import ProductRange, Product

  food_enabled = True
else:
  food_enabled = False

from alto_django_sk.models import BillHeader, BillItem

User = get_user_model()

@pytest.mark.skipif(not food_enabled, reason = 'needs food models')
class WriteBillItemTests(APITestCase, BaseAuthHelpers):
  def setUp(self):
    User.objects.create_superuser('admin', 'admin@test.cz', 'admin')

    self.admin_token = self.get_token(username = 'admin', password = 'admin').data['token']
    self.range1      = ProductRange.get_default()
    self.range2      = ProductRange.objects.create(name = 'other')
    self.product1    = self.range1.products.create(stock_card_number = 111,
                                                   name = '111', price = 100, tax_rate = 21)
    self.product2    = self.range1.products.create(stock_card_number = 222,
                                                   name = '222', price = 100, tax_rate = 21)
    self.bill_header = BillHeader.objects.create(c_kas = 1, c_uctu = 123)

  def test_with_product_id(self):
    item_data = {'header':     self.bill_header.id,
                 'product':    self.product1.id,
                 'c_stredisk': 123,
                 'dan':        '16.8',
                 'dan_sazba':  21,
                 'mnozstvi':   3,
                 'prachy':     '96.8',
                 'prachy_puv': 80}

    response = self.client.post(reverse('alto_django_sk:bill_items-list'), item_data,
                                **auth_header(self.admin_token))

    assert response.status_code == 201

    item = BillItem.objects.get()

    assert item.c_data  == self.range1
    assert item.c_karty == self.product1.stock_card_number

  def test_with_card_number(self):
    item_data = {'header':     self.bill_header.id,
                 'c_data':     self.range1.id,
                 'c_karty':    self.product1.stock_card_number,
                 'c_stredisk': 123,
                 'dan':        '16.8',
                 'dan_sazba':  21,
                 'mnozstvi':   3,
                 'prachy':     '96.8',
                 'prachy_puv': 80}

    response = self.client.post(reverse('alto_django_sk:bill_items-list'), item_data,
                                **auth_header(self.admin_token))

    assert response.status_code == 201

    item = BillItem.objects.get()

    assert item.product == self.product1

  def test_with_neither(self):
    item_data = {'header':     self.bill_header.id,
                 'c_stredisk': 123,
                 'dan':        '16.8',
                 'dan_sazba':  21,
                 'mnozstvi':   3,
                 'prachy':     '96.8',
                 'prachy_puv': 80}

    response = self.client.post(reverse('alto_django_sk:bill_items-list'), item_data,
                                **auth_header(self.admin_token))

    assert response.status_code == 400

    assert 'Je třeba zadat' in response.data['detail'][0]
    assert 'Je třeba zadat' in response.data['product'][0]
    assert 'Je třeba zadat' in response.data['c_karty'][0]

  def test_with_neither_and_no_default_range(self):
    Product.objects.all().delete()
    ProductRange.objects.all().delete()

    item_data = {'header':     self.bill_header.id,
                 'c_stredisk': 123,
                 'dan':        '16.8',
                 'dan_sazba':  21,
                 'mnozstvi':   3,
                 'prachy':     '96.8',
                 'prachy_puv': 80}

    response = self.client.post(reverse('alto_django_sk:bill_items-list'), item_data,
                                **auth_header(self.admin_token))

    assert response.status_code == 400

    assert _.find(response.data['c_data'], lambda error: 'Je třeba zadat'             in error)
    assert _.find(response.data['c_data'], lambda error: 'výchozí hodnota neexistuje' in error)

  def test_with_nonexistent_product_id(self):
    item_data = {'header':     self.bill_header.id,
                 'product':    999,
                 'c_stredisk': 123,
                 'dan':        '16.8',
                 'dan_sazba':  21,
                 'mnozstvi':   3,
                 'prachy':     '96.8',
                 'prachy_puv': 80}

    response = self.client.post(reverse('alto_django_sk:bill_items-list'), item_data,
                                **auth_header(self.admin_token))

    assert response.status_code == 400

    assert 'not found' in response.data['product'][0] # TODO: translate

  def test_with_nonexistent_card_number(self):
    item_data = {'header':     self.bill_header.id,
                 'c_data':     self.range1.id,
                 'c_karty':    999,
                 'c_stredisk': 123,
                 'dan':        '16.8',
                 'dan_sazba':  21,
                 'mnozstvi':   3,
                 'prachy':     '96.8',
                 'prachy_puv': 80}

    response = self.client.post(reverse('alto_django_sk:bill_items-list'), item_data,
                                **auth_header(self.admin_token))

    assert response.status_code == 400

    assert 'neexistuje' in response.data['detail'][0]
    assert 'neexistuje' in response.data['c_data'][0]
    assert 'neexistuje' in response.data['c_karty'][0]

  def test_with_nonexistent_range_id(self):
    item_data = {'header':     self.bill_header.id,
                 'c_data':     999,
                 'c_karty':    111,
                 'c_stredisk': 123,
                 'dan':        '16.8',
                 'dan_sazba':  21,
                 'mnozstvi':   3,
                 'prachy':     '96.8',
                 'prachy_puv': 80}

    response = self.client.post(reverse('alto_django_sk:bill_items-list'), item_data,
                                **auth_header(self.admin_token))

    assert response.status_code == 400

    assert 'not found' in response.data['c_data'][0] # TODO: translate

  def test_with_product_id_not_matching_card_number(self):
    item_data = {'header':     self.bill_header.id,
                 'product':    self.product1.id,
                 'c_data':     self.range2.id,
                 'c_karty':    self.product2.stock_card_number,
                 'c_stredisk': 123,
                 'dan':        '16.8',
                 'dan_sazba':  21,
                 'mnozstvi':   3,
                 'prachy':     '96.8',
                 'prachy_puv': 80}

    response = self.client.post(reverse('alto_django_sk:bill_items-list'), item_data,
                                **auth_header(self.admin_token))

    assert response.status_code == 400

    assert "Neodpovídá" in response.data['c_data'][0]
    assert "Neodpovídá" in response.data['c_karty'][0]
