import pytest
from decimal import Decimal
from django.conf import settings
from django.urls import reverse
from django.contrib.auth import get_user_model
from rest_framework.test import APITestCase
from alto_django_base_auth.tests.helpers import BaseAuthHelpers, auth_header

if 'alto_django_food' in settings.INSTALLED_APPS:
  from alto_django_food.models import ProductRange, Product

  food_enabled = True
else:
  food_enabled = False

from alto_django_sk.models import BillHeader, BillItem, BillPayment
from alto_django_sk.serializers import (BillHeaderSerializer, BillItemSerializer,
                                        BillPaymentSerializer)

User = get_user_model()

@pytest.mark.skipif(not food_enabled, reason = 'needs food models')
class WriteBillTests(APITestCase, BaseAuthHelpers):
  def setUp(self):
    User.objects.create_superuser('admin', 'admin@test.cz', 'admin')

    self.admin_token = self.get_token(username = 'admin', password = 'admin').data['token']
    self.range1      = ProductRange.objects.create(name = 'range1')
    self.range2      = ProductRange.objects.create(name = 'range2')
    self.product1    = self.range1.products.create(stock_card_number = 111,
                                                   name = '111', price = 100, tax_rate = 21)
    self.product2    = self.range2.products.create(stock_card_number = 222,
                                                   name = '111', price = 100, tax_rate = 21)

  def test_with_card_number_and_range_id_in_header(self):
    header_data  = {'c_uctu':     123,
                    'c_kas':      123,
                    'c_data':     self.range2.id}
    items_data   = [{'c_karty':    self.product1.stock_card_number,
                     'c_data':     self.range1.id, # explicit
                     'c_stredisk': 123,
                     'dan':        '16.8',
                     'dan_sazba':  21,
                     'mnozstvi':   2,
                     'prachy':     '96.8',
                     'prachy_puv': 80},
                    {'c_karty':    self.product2.stock_card_number,
                     # c_data taken from header
                     'c_stredisk': 123,
                     'dan':        '16.8',
                     'dan_sazba':  21,
                     'mnozstvi':   3,
                     'prachy':     '96.8',
                     'prachy_puv': 80}]
    payments_data = [{'cena_pl':    '96.8',
                      'druh_pl':    'HOT'}]

    response = self.client.post(reverse('alto_django_sk:write-bill'),
                                {'header':   header_data,
                                 'items':    items_data,
                                 'payments': payments_data},
                                **auth_header(self.admin_token))

    assert response.status_code == 200

    header  = BillHeader.objects.first()
    item1   = BillItem.objects.get(mnozstvi = 2)
    item2   = BillItem.objects.get(mnozstvi = 3)
    payment = BillPayment.objects.first()

    assert header.c_uctu   == 123

    assert item1.product   == self.product1
    assert item1.c_data    == self.range1
    assert item1.c_karty   == self.product1.stock_card_number

    assert item2.product   == self.product2
    assert item2.c_data    == self.range2
    assert item2.c_karty   == self.product2.stock_card_number

    assert payment.cena_pl == Decimal('96.8')

    assert response.data['header']      == BillHeaderSerializer(header).data
    assert response.data['items'][0]    == BillItemSerializer(item1).data
    assert response.data['payments'][0] == BillPaymentSerializer(payment).data
