from pydash import py_
from django.conf import settings
from django.db import transaction
from django.utils.translation import gettext as _
from rest_framework.serializers import Serializer, ValidationError
from alto_django_utils.utils import add_error
from alto_django_utils.serializers import serializer_for

if 'alto_django_food' in settings.INSTALLED_APPS:
  from alto_django_food.models import ProductRange, Product

  food_enabled = True
else:
  food_enabled = False

from .models import Bill, BillClosing, BillHeader, BillItem, BillPayment, Sequence

BillClosingSerializer = serializer_for(BillClosing)
BillHeaderSerializer  = serializer_for(BillHeader)
BaseItemSerializer    = serializer_for(BillItem)
BillPaymentSerializer = serializer_for(BillPayment)
SequenceSerializer    = serializer_for(Sequence)

class BillItemSerializer(BaseItemSerializer):
  def __init__(self, *args, **kwargs):
    super().__init__(*args, **kwargs)

    if 'product' in self.fields: self.fields['product'].required = False
    if 'c_data'  in self.fields: self.fields['c_data' ].required = False
    if 'c_karty' in self.fields: self.fields['c_karty'].required = False

  def validate(self, data):
    errors = {}

    if 'product' in data:
      self._product_given(data, errors)
    else:
      if food_enabled and not 'c_data' in data:
        default_product_range = ProductRange.get_default()

        if default_product_range: data['c_data'] = default_product_range
        else: add_error(errors, 'c_data', _("Wasn't given and default doesn't exist."))

      if 'c_data' in data and 'c_karty' in data:
        self._range_and_card_number_given(data, errors)
      else:
        self._neither_given(data, errors)

    if errors: raise ValidationError(errors)

    return data

  def _product_given(self, data, errors):
    product = data['product']

    if 'c_data'  in data and data['c_data']  != product.range:
      add_error(errors, 'c_data', _("Doesn't match product.range."))

    if 'c_karty' in data and data['c_karty'] != product.stock_card_number:
      add_error(errors, 'c_karty', _("Doesn't match product.stock_card_number."))

    data['c_data']  = product.range
    data['c_karty'] = product.stock_card_number

  def _range_and_card_number_given(self, data, errors):
    if not food_enabled: return

    product_range = data['c_data']
    card_number   = data['c_karty']

    try:
      data['product'] = product_range.products.get(stock_card_number = card_number)
    except Product.DoesNotExist:
      params  = {'range_id': product_range.id, 'stock_card_number': card_number}
      message = _("Product with range_id = %(range_id)s and stock_card_number = "
                  "%(stock_card_number)s doesn't exist.") % params

      add_error(errors, ['detail', 'c_data', 'c_karty'], message)

  def _neither_given(self, data, errors):
    message = _("You must provide either 'product' or 'c_data' and 'c_karty'.")
    add_error(errors, ['detail', 'product'], message)

    if not 'c_data'  in data: add_error(errors, 'c_data',  message)
    if not 'c_karty' in data: add_error(errors, 'c_karty', message)

class NestedBillItemSerializer(BillItemSerializer):
  class Meta(BillItemSerializer.Meta):
    read_only_fields = ('header',)

class NestedBillPaymentSerializer(BillPaymentSerializer):
  class Meta(BillPaymentSerializer.Meta):
    read_only_fields = ('header',)

class BillSerializer(Serializer):
  header   = BillHeaderSerializer()
  items    = NestedBillItemSerializer(   many = True, allow_empty = False)
  payments = NestedBillPaymentSerializer(many = True, allow_empty = False)

  def __init__(self, *args, **kwargs):
    super().__init__(*args, **kwargs)

    header = self.initial_data.get('header')
    items  = self.initial_data.get('items')

    if (header and isinstance(header, dict) and 'c_data' in header
        and items and isinstance(items, list) and all(isinstance(item, dict) for item in items)):
      c_data = header['c_data']
      self.initial_data['items'] = [py_.defaults(item, {'c_data': c_data}) for item in items]

  @transaction.atomic
  def create(self, validated_data):
    header_serializer = BillHeaderSerializer(data = validated_data['header'])
    header_serializer.is_valid(raise_exception = True)

    header   = header_serializer.save()
    defaults = {'header': header.id}

    items_data = [py_.defaults(item, defaults) for item in validated_data['items']]
    items_serializer = BillItemSerializer(many = True, data = items_data)
    items_serializer.is_valid(raise_exception = True)
    items = items_serializer.save()

    payments_data = [py_.defaults(payment, defaults) for payment in validated_data['payments']]
    payments_serializer = BillPaymentSerializer(many = True, data = payments_data)
    payments_serializer.is_valid(raise_exception = True)
    payments = payments_serializer.save()

    return Bill(header, items, payments)
