from copy import deepcopy
from django.utils.translation import gettext as _
from rest_framework.views import APIView
from rest_framework.response import Response
from rest_framework.exceptions import ValidationError
from alto_django_utils.params import WithParams
from alto_django_utils.exceptions import MailSendingFailed
from alto_django_utils.permissions import AllowAny
from alto_django_kredit.settings import settings
from alto_django_client_auth.services.registration import (RegisterClientUser,
                                                           RegisterClientUserWithBV,
                                                           SendActivationEmail)

# TODO: translate messages properly
class RegisterClient(APIView, WithParams):
  permission_classes = [AllowAny]
  params_spec = { 'query': { 'required': ['card_number', 'redirect'] },
                  'body':  { 'required': ['password'],
                             'optional': ['email', 'username', 'use_email_as_username'] } }

  def post(self, request):
    params = self.params(request)

    try:
      data = self.call_service(params, request.build_absolute_uri)

      return Response({ 'user': data })
    except RegisterClientUser.InvalidCardNumber:
      self.invalid_card_number(params.card_number)
    except RegisterClientUser.ClientNotFound:
      self.client_not_found(params.card_number)
    except RegisterClientUser.MultipleClientsFound:
      self.multiple_clients_found(params.card_number)
    except RegisterClientUser.ClientAlreadyRegistered:
      self.client_already_registered(params.card_number)
    except RegisterClientUser.EmailAlreadyRegistered as error:
      self.email_already_registered(error.email)
    except RegisterClientUser.ValidationError as error:
      raise ValidationError(error.errors)
    except SendActivationEmail.SendingFailed as error:
      raise MailSendingFailed(detail = _("Activation email couldn't be sent"),
                              additional_detail = str(error.__cause__))

  def call_service(self, params, uri_builder):
    service = RegisterClientUser(params, uri_builder)

    return service.perform()

  def invalid_card_number(self, card_number):
    raise ValidationError({'card_number': [f'Invalid card_number: {card_number}'],
                           'code': 'invalid_card_number'})

  def client_not_found(self, card_number):
    raise ValidationError({'card_number': f'Klient s číslem karty {card_number} nebyl nalezen.',
                           'code': 'client_not_found'})

  def multiple_clients_found(self, card_number):
    raise ValidationError({'card_number': f'Nalezeno několik klientů s číslem karty {card_number}.',
                           'code': 'multiple_clients_found'})

  def client_already_registered(self, card_number):
    raise ValidationError({'card_number': f'Klient s číslem karty {card_number} už je registrován.',
                           'code': 'client_already_registered'})

  def email_already_registered(self, email):
    raise ValidationError({'email': f"Uživatel s emailem '{email}' už je registrován.",
                           'code': 'email_already_registered'})

class RegisterClientWithBV(RegisterClient):
  params_spec = deepcopy(RegisterClient.params_spec)
  params_spec['body']['required'].extend(['bill_number', 'bill_date'])

  def call_service(self, params, uri_builder):
    service = RegisterClientUserWithBV(params, uri_builder)

    try:
      return service.perform()
    except RegisterClientUserWithBV.BillNotFound:
      self.bill_not_found()
    except RegisterClientUserWithBV.MultipleBillsFound:
      self.multiple_bills_found(params.bill_number, params.bill_date)
    except RegisterClientUserWithBV.BillDoesNotBelongToClient:
      self.bill_does_not_belong_to_client()

  def bill_not_found(self):
    message = 'Doklad nebyl nalezen'
    raise ValidationError({ 'bill_number': [message], 'bill_date': [message],
                            'code': 'bill_not_found' })

  def multiple_bills_found(self, bill_number, bill_date):
    message = f'Nalezeno několik dokladů s číslem {bill_number} a datem {bill_date}.'
    raise ValidationError({ 'bill_number': [message], 'bill_date': [message],
                            'code': 'multiple_bills_found' })

  def bill_does_not_belong_to_client(self):
    message = 'Doklad nepatří klientu s uvedeným číslem karty'
    raise ValidationError({ 'bill_number': [message], 'bill_date': [message],
                            'code': 'bill_does_not_belong_to_client' })
