from django.shortcuts import get_object_or_404
from django.utils.translation import gettext as _
from rest_framework.response import Response
from rest_framework.exceptions import NotFound
from rest_framework.decorators import api_view, permission_classes
from alto_django_utils.params import required_query_param, optional_query_param
from alto_django_utils.permissions import IsAuthenticated, IsAdminOrStaff
from alto_django_kredit.models import Client
from alto_django_client_auth.models import User

def to_bool(string):
  return string and (string in ['1', 't', 'T'] or string.lower() == 'true')

def clear_sensitive_data(client):
  client.email    = None
  client.personal = {}
  client.company  = {}
  client.address  = {}
  client.address2 = {}

  client.save()

@api_view(['DELETE'])
@permission_classes([IsAuthenticated])
def delete_user(request):
  clear_sensitive = to_bool(optional_query_param(request, 'clear_sensitive_data'))

  if clear_sensitive:
    client = request.user.client

    if client:
      clear_sensitive_data(client)
    else:
      message = _('You asked to clear sensitive data, but this is not a client user')
      return Response({ 'detail': message }, status = 400)

  request.user.delete()

  return Response(status = 204)

@api_view(['DELETE'])
@permission_classes([IsAdminOrStaff])
def delete_client_user(request):
  client_id       = required_query_param(request, 'client_id')
  client          = get_object_or_404(Client, pk = client_id)
  clear_sensitive = to_bool(optional_query_param(request, 'clear_sensitive_data'))

  try:
    client.user.delete()

    if clear_sensitive:
      clear_sensitive_data(client)

    return Response(status = 204)
  except User.DoesNotExist:
    message = _("Client with id %(id)s isn't registered.") % {'id': client_id }

    raise NotFound({ 'detail': message })
