from django.urls import reverse
from django.contrib.auth import get_user_model
from rest_framework.test import APITestCase
from alto_django_settings.models import Setting
from alto_django_kredit.tests.factories.client import ClientFactory
from alto_django_client_auth.tests.helpers import ClientAuthHelpers, auth_header

User = get_user_model()

class IsProfileCompleteTests(APITestCase, ClientAuthHelpers):
  def test_basic(self):
    Setting.objects.create(key   = 'auth.profile.required_fields',
                           value = ['firstname', 'surname', 'email'])

    client   = ClientFactory.create(firstname = None)
    user     = User.objects.create_user('test', 'test@test.cz', 'test', client = client)
    token    = self.get_token(username = 'test', password = 'test').data['token']

    response = self.client.get(reverse('alto_django_client_auth:is-profile-complete'),
                                **auth_header(token))

    assert response.data['is_complete'] is False

    client.firstname = 'Test'
    client.save()

    response = self.client.get(reverse('alto_django_client_auth:is-profile-complete'),
                                **auth_header(token))

    assert response.data['is_complete'] is True

  def test_nested_fields(self):
    Setting.objects.create(key   = 'auth.profile.required_fields',
                           value = ['personal.birth_date'])

    client   = ClientFactory.create(personal = {'phone': '602222222'})
    user     = User.objects.create_user('test', 'test@test.cz', 'test', client = client)
    token    = self.get_token(username = 'test', password = 'test').data['token']

    response = self.client.get(reverse('alto_django_client_auth:is-profile-complete'),
                                **auth_header(token))

    assert response.data['is_complete'] is False

    client.personal = {'birth_date': '2000-01-01'}
    client.save()

    response = self.client.get(reverse('alto_django_client_auth:is-profile-complete'),
                                **auth_header(token))

    assert response.data['is_complete'] is True

  def test_non_client_user(self):
    Setting.objects.create(key   = 'auth.profile.required_fields',
                           value = ['personal.birth_date'])

    user     = User.objects.create_user('test', 'test@test.cz', 'test')
    token    = self.get_token(username = 'test', password = 'test').data['token']

    response = self.client.get(reverse('alto_django_client_auth:is-profile-complete'),
                                **auth_header(token))

    assert response.status_code    == 404
    assert response.data['detail'] == 'Uživatel test není asociován s žádným klientem.'

  def test_missing_setting(self):
    client   = ClientFactory.create(personal = {'phone': '602222222'})
    user     = User.objects.create_user('test', 'test@test.cz', 'test', client = client)
    token    = self.get_token(username = 'test', password = 'test').data['token']

    response = self.client.get(reverse('alto_django_client_auth:is-profile-complete'),
                                **auth_header(token))

    # defaults to no required fields
    assert response.status_code         == 200
    assert response.data['is_complete'] is True

  def test_invalid_setting(self):
    client   = ClientFactory.create(personal = {'phone': '602222222'})
    user     = User.objects.create_user('test', 'test@test.cz', 'test', client = client)
    token    = self.get_token(username = 'test', password = 'test').data['token']

    assert self._fails_setting_validation('non-array', token)
    assert self._fails_setting_validation([1, 2, 3], token)
    assert self._fails_setting_validation(['!@#'], token)
    assert self._fails_setting_validation(['123'], token)
    assert self._fails_setting_validation(['...'], token)

  def test_nonexistent_field_segment(self):
    client   = ClientFactory.create(personal = {'phone': '602222222'})
    user     = User.objects.create_user('test', 'test@test.cz', 'test', client = client)
    token    = self.get_token(username = 'test', password = 'test').data['token']

    assert self._fails_to_find_field('nonexistent',                token)
    assert self._fails_to_find_field('personal.nonexistent',       token)
    assert self._fails_to_find_field('personal.phone.nonexistent', token)

  def _fails_setting_validation(self, setting, token):
    Setting.objects.all().delete()
    Setting.objects.create(key   = 'auth.profile.required_fields',
                           value = setting)

    response = self.client.get(reverse('alto_django_client_auth:is-profile-complete'),
                                **auth_header(token))

    message  = 'auth.profile.required_fields setting must be a list of dotted string paths'

    return (response.status_code    == 412 and
            response.data['detail'] == message)

  def _fails_to_find_field(self, field, token):
    Setting.objects.all().delete()
    Setting.objects.create(key   = 'auth.profile.required_fields',
                           value = [field])

    response = self.client.get(reverse('alto_django_client_auth:is-profile-complete'),
                                **auth_header(token))

    return (response.data['is_complete'] is False and
            response.data['detail']      == 'Missing field ' + field)
