from datetime import date
from django.contrib.auth import get_user_model
from rest_framework.test import APITestCase
from alto_django_kredit.tests.factories.client import ClientFactory
from alto_django_client_auth.tests.helpers import ClientAuthHelpers

User = get_user_model()

class EditClientTests(APITestCase, ClientAuthHelpers):
  def test_updates_client_attributes(self):
    client = ClientFactory.create(firstname = 'test', surname = 'test')
    user   = User.objects.create_user('test', 'test@test.cz', 'test', client = client)
    token  = self.get_token(username = 'test', password = 'test').data['token']

    response = self.edit_client({ 'firstname': 'changed', 'surname': 'changed' }, token)

    client.refresh_from_db()

    assert response.status_code == 200
    assert client.firstname     == 'changed'
    assert client.surname       == 'changed'

  def test_updates_birth_date_updated_at(self):
    client = ClientFactory.create(firstname = 'test', surname = 'test',
                                  personal = { 'birth_date': '2001-01-01' })
    user   = User.objects.create_user('test', 'test@test.cz', 'test', client = client)
    token  = self.get_token(username = 'test', password = 'test').data['token']

    response = self.edit_client({ 'firstname': 'changed', 'surname': 'changed',
                                  'personal': { 'birth_date': '2002-02-02' } }, token)

    client.refresh_from_db()

    assert response.status_code                     == 200
    assert client.personal['birth_date_updated_at'] == date.today().isoformat()

  def test_updates_user(self):
    client = ClientFactory.create(firstname = 'test', surname = 'test')
    user   = User.objects.create_user('test', 'test@test.cz', 'test', client = client)
    token  = self.get_token(username = 'test', password = 'test').data['token']

    response = self.edit_client({ 'firstname': 'changed', 'surname': 'changed',
                                  'email': 'changed@test.cz' }, token)

    client.refresh_from_db()
    user.refresh_from_db()

    assert response.status_code == 200
    assert client.email         == 'changed@test.cz'
    assert user.email           == 'changed@test.cz'
    assert user.first_name      == 'changed'
    assert user.last_name       == 'changed'
    assert user.username        == 'test' # not changed

  def test_updates_user_login_if_same_as_email(self):
    client = ClientFactory.create(firstname = 'test', surname = 'test')
    user   = User.objects.create_user('test@test.cz', 'test@test.cz', 'test', client = client)
    token  = self.get_token(username = 'test@test.cz', password = 'test').data['token']

    response = self.edit_client({ 'email': 'changed@test.cz' }, token)

    client.refresh_from_db()
    user.refresh_from_db()

    assert response.status_code == 200
    assert user.username        == 'changed@test.cz'

  def test_user_with_email_or_username_already_exists(self):
    client = ClientFactory.create(firstname = 'test', surname = 'test')
    user   = User.objects.create_user('test@test.cz', 'test@test.cz', 'test', client = client)
    user   = User.objects.create_user('test2@test.cz', 'test2@test.cz', 'test')
    token  = self.get_token(username = 'test@test.cz', password = 'test').data['token']

    response = self.edit_client({ 'email': 'test2@test.cz' }, token)

    assert response.status_code              == 400
    assert response.data['user']['email']    == ['Uživatel s tímto emailem již existuje.']
    assert response.data['user']['username'] == ['Uživatel s tímto uživatelským jménem již existuje.']

  def test_non_client_user(self):
    user  = User.objects.create_user('test', 'test@test.cz', 'test')
    token = self.get_token(username = 'test', password = 'test').data['token']

    response = self.edit_client({ 'firstname': 'changed', 'surname': 'changed' }, token)

    assert response.status_code    == 404
    assert response.data['detail'] == 'Uživatel test není asociován s žádným klientem.'
