import django.core.exceptions
from django.db import transaction
from django.core.exceptions import ObjectDoesNotExist
from django.contrib.auth import get_user_model
from django.contrib.auth.password_validation import validate_password
from alto_django_utils.shortcuts import get_object_or_none
from alto_django_base_auth.serializers import UserSerializer
from alto_django_base_auth.services import SendActivationEmail
from alto_django_kredit.settings import settings
from alto_django_kredit.models import Client

User = get_user_model()

class RegisterClientUser:
  InvalidCardNumber    = settings.client_finder.InvalidCardNumber
  ClientNotFound       = settings.client_finder.ClientNotFound
  MultipleClientsFound = settings.client_finder.MultipleClientsFound

  class ClientAlreadyRegistered(Exception): pass

  class EmailAlreadyRegistered(Exception):
    def __init__(self, email):
      self.email = email

  class ValidationError(Exception):
    def __init__(self, errors):
      self.errors = errors

  def __init__(self, params, uri_builder):
    self.params      = params
    self.uri_builder = uri_builder

  # @raise InvalidCardNumber
  # @raise ClientNotFound
  # @raise MultipleClientsFound
  # @raise ClientAlreadyRegistered
  # @raise EmailAlreadyRegistered
  # @raise ValidationError
  def perform(self):
    client = self._find_client()

    self._validate_preconditions(client)

    with transaction.atomic():
      serializer = self._validate_and_save_both(client)

      SendActivationEmail(serializer.instance, self.params.redirect, self.uri_builder)()

    return serializer.data

  # @raise InvalidCardNumber
  # @raise ClientNotFound
  # @raise MultipleClientsFound
  def _find_client(self):
    try:
      return settings.client_finder.clients_for_card(self.params.card_number).get()
    except Client.DoesNotExist:
      raise self.ClientNotFound()
    except Client.MultipleObjectsReturned:
      raise self.MultipleClientsFound()

  # @raise ClientAlreadyRegistered
  # @raise EmailAlreadyRegistered
  def _validate_preconditions(self, client):
    self._ensure_client_not_registered(client)
    self._ensure_email_not_registered(client)

  # @raise ClientAlreadyRegistered
  def _ensure_client_not_registered(self, client):
    try:
      if client.user:
        raise self.ClientAlreadyRegistered()
    except ObjectDoesNotExist: pass

  # @raise EmailAlreadyRegistered
  def _ensure_email_not_registered(self, client):
    email = self.params.email or client.email

    if get_object_or_none(User, email = email):
      raise self.EmailAlreadyRegistered(email)

  # @raise ValidationError
  def _validate_and_save_both(self, client):
    attributes = self._user_attributes_for(client)
    serializer = UserSerializer(data = attributes)

    self._validate_user(serializer)

    user = serializer.save()

    self._validate_password(user, self.params.password)
    user.set_password(self.params.password)

    client.user = user
    if self.params.email: client.email = self.params.email

    user.save()
    client.save()

    return serializer

  # @raise ValidationError
  def _validate_user(self, serializer):
    if not serializer.is_valid():
      raise self.ValidationError(serializer.errors)

  # @raise ValidationError
  def _validate_password(self, user, password):
    try:
      validate_password(password, user)
    except django.core.exceptions.ValidationError as error:
      raise self.ValidationError({ 'password': error.messages })

  def _user_attributes_for(self, client):
    username = (self.params.email or client.email if self.params.use_email_as_username
                else self.params.username)

    return {
      'username':   username,
      'first_name': client.firstname,
      'last_name':  client.surname,
      'email':      self.params.email or client.email,
      'is_active':  False }

class RegisterClientUserWithBV(RegisterClientUser):
  BillNotFound              = settings.client_finder.BillNotFound
  MultipleBillsFound        = settings.client_finder.MultipleBillsFound
  BillDoesNotBelongToClient = settings.client_finder.BillDoesNotBelongToClient

  # @raise InvalidCardNumber
  # @raise ClientNotFound
  # @raise BillNotFound
  # @raise MultipleBilssFound
  # @raise BillDoesNotBelongToClient
  def _find_client(self):
    return settings.client_finder \
      .client_for_card_and_bill(self.params.card_number,
                                self.params.bill_number,
                                self.params.bill_date)
