from alto_django_settings.models import Setting

class EnsureProfileComplete:
  class InvalidRequiredFieldsSetting(Exception):
    def __init__(self):
      message = 'auth.profile.required_fields setting must be a list of dotted string paths'

      super().__init__(message)

  class MissingField(Exception):
    def __init__(self, field):
      super().__init__('Missing field ' + field)

  class MissingValue(Exception):
    def __init__(self, field):
      super().__init__('Missing value for ' + field)


  def __init__(self, client):
    self.client          = client
    self.required_fields = Setting.for_key('auth.profile.required_fields', [])

    if not self._is_valid_setting(self.required_fields):
      raise self.InvalidRequiredFieldsSetting()

  def _is_valid_setting(self, setting):
    return (isinstance(setting, list) and
            all(self._is_valid_field(field) for field in setting))

  def _is_valid_field(self, field):
    return (isinstance(field, str) and \
            all(segment.isidentifier() for segment in field.split('.')))


  def __call__(self):
    for field in self.required_fields:
      value = self._get_field_value(self.client, field)

      if not (value or value == 0):
        raise self.MissingValue(field)

  def _get_field_value(self, client, field):
    segments = field.split('.')

    try:
      value = getattr(client, segments[0])
    except AttributeError:
      raise self.MissingField(field)

    for segment in segments[1:]:
      if isinstance(value, dict):
        try:
          value = value[segment]
        except KeyError:
          raise self.MissingField(field)
      else:
        raise self.MissingField(field)

    return value
