from rest_framework.exceptions import ValidationError, MethodNotAllowed
from alto_django_utils.viewsets import viewset_for
from alto_django_utils.permissions import IsAuthenticated
from alto_django_kredit.models import Client, ClientInfo, Transaction
from alto_django_kredit.queries import TransactionRecomputer
from alto_django_kredit.serializers import (ClientSerializer, ClientInfoSerializer,
                                            TransactionSerializer, HistoricTransactionSerializer)

ClientViewSet          = viewset_for(Client,      ClientSerializer)
ClientInfoViewSet      = viewset_for(ClientInfo,  ClientInfoSerializer)
BaseTransactionViewSet = viewset_for(Transaction, TransactionSerializer)

# this expects AUTH_USER_MODEL to be alto_django_client_auth.User, but this project doesn't strictly
# depend on alto-django-client-auth, because that would introduce a circular dependency
# -> use this only in projects which use alto_django_client_auth.User as its AUTH_USER_MODEL
class TransactionViewSet(BaseTransactionViewSet):
  permission_classes = [IsAuthenticated]

  def get_queryset(self):
    user = self.request.user

    if user.is_superuser or user.is_staff:
      return super().get_queryset()
    elif getattr(user, 'client', None):
      return user.client.transactions
    else:
      message = 'Authenticated user is neither admin, nor staff, nor client user'
      raise ValidationError({ 'detail': message })

  def update(self, request, *args, **kwargs):
    raise MethodNotAllowed(request.method, 'Changing transactions is not allowed.')

  def destroy(self, request, *args, **kwargs):
    raise MethodNotAllowed(request.method, 'Deleting transactions is not allowed.')

class HistoricTransactionViewSet(BaseTransactionViewSet):
  permission_classes = [IsAuthenticated]
  serializer_class   = HistoricTransactionSerializer

  def perform_destroy(self, instance):
    with TransactionRecomputer(instance.client, instance.kind):
      instance.delete()
