from django.urls import reverse
from django.contrib.auth import get_user_model
from rest_framework.test import APITestCase
from alto_django_kredit.models import Client, ClientInfo
from alto_django_kredit.services import CreateTransaction
from alto_django_client_auth.tests.helpers import ClientAuthHelpers, auth_header

User = get_user_model()

class TransactionSumsTests(APITestCase, ClientAuthHelpers):
  def setUp(self):
    self.client1 = Client.objects.create(client_id = '123', card_id = '123',
                                         firstname = 'Tomáš', surname = 'Marný',
                                         email = 'client1@test.cz',
                                         personal = { 'sex': 'M', 'birth_date': '2000-01-01' })
    self.client2 = Client.objects.create(client_id = '234', card_id = '234',
                                         email = 'client2@test.cz')
    self.client3 = Client.objects.create(client_id = '345', card_id = '345',
                                         email = 'client3@test.cz')

    ClientInfo.objects.create(client = self.client1, typ = 'T')

    CreateTransaction(client = self.client1, # before start
                      kind = 'KC',
                      amount = -100,
                      transacted_at = '2010-03-31T23:59:59.999Z')()
    CreateTransaction(client = self.client1, # OK
                      kind = 'KC',
                      amount = -100,
                      transacted_at = '2010-04-01T00:00Z')()
    CreateTransaction(client = self.client1, # wrong kind
                      kind = 'POI',
                      amount = -100,
                      transacted_at = '2010-04-10T12:00Z')()
    CreateTransaction(client = self.client1, # not negative
                      kind = 'KC',
                      amount = 100,
                      transacted_at = '2010-04-10T12:00Z')()
    CreateTransaction(client = self.client1, # OK
                      kind = 'KC',
                      amount = -100,
                      transacted_at = '2010-04-30T23:59:59.999Z')()
    CreateTransaction(client = self.client1, # after end
                      kind = 'KC',
                      amount = -100,
                      transacted_at = '2010-05-01T00:00Z')()

    CreateTransaction(client = self.client3, # OK, but high sum
                      kind = 'KC',
                      amount = -10000,
                      transacted_at = '2010-04-10T12:00Z')()

    User.objects.create_user('admin', 'admin@test.cz', 'admin', is_superuser = True)
    self.token = self.get_token(username = 'admin', password = 'admin').data['token']

  def test_simple(self):
    path     = reverse('alto_django_kredit:clients-with-transaction-sums')
    start    = '2010-04-01T00:00Z'
    end      = '2010-05-01T00:00Z'
    kind     = 'KC'

    response = self.client.get(f'{path}?start={start}&end={end}&kind={kind}',
                               **auth_header(self.token))
    results  = response.data['results']

    self.assertEqual(response.status_code, 200)
    self.assertEqual(response.data['count'], 3)

    self.assertEqual(results[0]['id'],         self.client1.id)
    self.assertEqual(results[0]['email'],      self.client1.email)
    self.assertEqual(results[0]['firstname'],  self.client1.firstname)
    self.assertEqual(results[0]['surname'],    self.client1.surname)
    self.assertEqual(results[0]['typ'],        self.client1.info.typ)
    self.assertEqual(results[0]['sex'],        self.client1.personal['sex'])
    self.assertEqual(results[0]['birth_date'], self.client1.personal['birth_date'])
    self.assertEqual(results[0]['sum'],        '200.00')

    self.assertEqual(results[1]['id'],         self.client2.id)
    self.assertEqual(results[1]['email'],      self.client2.email)
    self.assertEqual(results[1]['firstname'],  None)
    self.assertEqual(results[1]['surname'],    None)
    self.assertEqual(results[1]['typ'],        None)
    self.assertEqual(results[1]['sex'],        None)
    self.assertEqual(results[1]['birth_date'], None)
    self.assertEqual(results[1]['sum'],        '0')

    self.assertEqual(results[2]['id'],         self.client3.id)
    self.assertEqual(results[2]['sum'],        '10000.00')

  def test_sum_bounds(self):
    path     = reverse('alto_django_kredit:clients-with-transaction-sums')
    start    = '2010-04-01T00:00Z'
    end      = '2010-05-01T00:00Z'
    kind     = 'KC'
    min_sum  = 100
    max_sum  = 1000

    response = self.client.get(f'{path}?start={start}&end={end}&kind={kind}&'
                               f'min_sum={min_sum}&max_sum={max_sum}',
                               **auth_header(self.token))

    self.assertEqual(response.status_code, 200)
    self.assertEqual(response.data['count'], 1)
    self.assertEqual(response.data['results'][0]['id'], self.client1.id)

  def test_pagination(self):
    path     = reverse('alto_django_kredit:clients-with-transaction-sums')
    start    = '2010-04-01T00:00Z'
    end      = '2010-05-01T00:00Z'
    kind     = 'KC'

    response1 = self.client.get(f'{path}?start={start}&end={end}&kind={kind}&page=1&page_size=1',
                                **auth_header(self.token))
    response2 = self.client.get(f'{path}?start={start}&end={end}&kind={kind}&page=2&page_size=1',
                                **auth_header(self.token))
    response3 = self.client.get(f'{path}?start={start}&end={end}&kind={kind}&page=3&page_size=1',
                                **auth_header(self.token))

    self.assertEqual(response1.data['count'], 3)
    self.assertEqual(len(response1.data['results']), 1)
    self.assertEqual(response1.data['results'][0]['id'], self.client1.id)

    self.assertEqual(response2.data['count'], 3)
    self.assertEqual(len(response2.data['results']), 1)
    self.assertEqual(response2.data['results'][0]['id'], self.client2.id)

    self.assertEqual(response3.data['count'], 3)
    self.assertEqual(len(response3.data['results']), 1)
    self.assertEqual(response3.data['results'][0]['id'], self.client3.id)

  def test_bad_params(self):
    path     = reverse('alto_django_kredit:clients-with-transaction-sums')
    start    = '2010-04-01T00:00Z'
    end      = '2010-05-01T00:00Z'
    kind     = 'KC'

    response = self.client.get(f'{path}?start=bad&end={end}&kind={kind}',
                               **auth_header(self.token))

    self.assertEqual(response.status_code, 400)
    self.assertIn('Chybný formát data a času', response.data['start'][0])

    response = self.client.get(f'{path}?start={start}&end=bad&kind={kind}',
                               **auth_header(self.token))

    self.assertEqual(response.status_code, 400)
    self.assertIn('Chybný formát data a času', response.data['end'][0])

    response = self.client.get(f'{path}?start={start}&end={end}&kind={kind}&min_sum=bad',
                               **auth_header(self.token))

    self.assertEqual(response.status_code, 400)
    self.assertIn('Je vyžadováno číslo', response.data['min_sum'][0])

    response = self.client.get(f'{path}?start={start}&end={end}&kind={kind}&max_sum=bad',
                               **auth_header(self.token))

    self.assertEqual(response.status_code, 400)
    self.assertIn('Je vyžadováno číslo', response.data['max_sum'][0])

    response = self.client.get(f'{path}?start={start}&end={end}&kind={kind}&page=bad&page_size=10',
                               **auth_header(self.token))

    self.assertEqual(response.status_code, 400)
    self.assertIn('Je vyžadováno celé číslo', response.data['page'][0])

    response = self.client.get(f'{path}?start={start}&end={end}&kind={kind}&page=1&page_size=bad',
                               **auth_header(self.token))

    self.assertEqual(response.status_code, 400)
    self.assertIn('Je vyžadováno celé číslo', response.data['page_size'][0])
