from django.test import TestCase
from alto_django_kredit.models import Client, Transaction
from alto_django_kredit.queries.transaction_recomputer import TransactionRecomputer

def refresh(entity):
  entity.refresh_from_db()

  return entity

class RecomputeTransactionBalancesTests(TestCase):
  def setUp(self):
    self.client1 = Client.objects.create(client_id = 123, card_id = 123)
    self.client2 = Client.objects.create(client_id = 234, card_id = 234)

    self.transactions = Transaction.objects.bulk_create([
      Transaction(client = self.client1, kind = 'KC', amount = -300, balance = -300),
      Transaction(client = self.client1, kind = 'KC', amount =  300, balance = 0, description = 'VYUCT'),
      Transaction(client = self.client1, kind = 'KC', amount = -100, balance = -100),
      Transaction(client = self.client1, kind = 'KC', amount = -100, balance = -200),
      Transaction(client = self.client1, kind = 'KC', amount = -100, balance = -300),
      Transaction(client = self.client1, kind = 'KC', amount =  300, balance = 0, description = 'VYUCT'),
      Transaction(client = self.client1, kind = 'KC', amount = -300, balance = -300),
      Transaction(client = self.client1, kind = 'SK', amount = -300, balance = -300),
      Transaction(client = self.client2, kind = 'SK', amount = -300, balance = -300)
    ])

  def test_change_normal_transaction_balance(self):
    transaction = self.transactions[3]

    transaction.balance = 1000
    transaction.save()

    TransactionRecomputer()()

    assert refresh(transaction).amount  == -100 # preserved amount
    assert refresh(transaction).balance == -200 # fixed balance

  def test_change_normal_transaction_amount(self):
    transaction = self.transactions[3]

    transaction.amount = -200
    transaction.save()

    TransactionRecomputer()()

    assert refresh(self.transactions[3]).balance == -300 # fixed balance
    assert refresh(self.transactions[4]).balance == -400 # fixed balance
    assert refresh(self.transactions[5]).balance ==    0 # preserved balance
    assert refresh(self.transactions[5]).amount  ==  400 # fixed amount

  def test_change_settlement_transaction_amount(self):
    transaction = self.transactions[1]

    transaction.amount = 1000
    transaction.save()

    TransactionRecomputer()()

    assert refresh(transaction).amount  == 300 # fixed amount
    assert refresh(transaction).balance == 0   # preserved balance

  def test_change_settlement_transaction_balance(self):
    transaction = self.transactions[1]

    transaction.balance = 1000
    transaction.save()

    TransactionRecomputer()()

    assert refresh(self.transactions[1]).balance == 1000 # preserved balance
    assert refresh(self.transactions[1]).amount  == 1300 # fixed amount
    assert refresh(self.transactions[2]).balance ==  900 # fixed balance
    assert refresh(self.transactions[2]).amount  == -100 # preserved amount
    assert refresh(self.transactions[3]).balance ==  800 # fixed balance
    assert refresh(self.transactions[3]).amount  == -100 # preserved amount
    assert refresh(self.transactions[4]).balance ==  700 # fixed balance
    assert refresh(self.transactions[4]).amount  == -100 # preserved amount
    assert refresh(self.transactions[5]).balance ==    0 # preserved balance
    assert refresh(self.transactions[5]).amount  == -700 # fixed amount

  def test_fix_only_transactions_of_one_client_and_kind(self):
    transaction1 = self.transactions[3]
    transaction2 = self.transactions[7]
    transaction3 = self.transactions[8]

    transaction1.balance = 1000
    transaction2.balance = 1000
    transaction3.balance = 1000

    transaction1.save()
    transaction2.save()
    transaction3.save()

    TransactionRecomputer(client = self.client1, kind = 'KC')()

    assert refresh(transaction1).balance == -200 # fixed
    assert refresh(transaction2).balance == 1000 # not fixed
    assert refresh(transaction3).balance == 1000 # not fixed
