from django.utils import timezone
from django.db.models import (
  Model, CharField, BooleanField, DateField, DateTimeField, IntegerField, SmallIntegerField,
  DecimalField, BigAutoField, ForeignKey, OneToOneField, CASCADE, Q
)
from django.contrib.postgres.fields import JSONField
from alto_django_utils.models import PROTECT
from alto_django_sk.models import BillHeader

class Client(Model):
  class Meta:
    db_table            = 'clients'
    default_permissions = ('view', 'add', 'change', 'delete')
    verbose_name        = 'klient'
    verbose_name_plural = 'klienti'

  status          = CharField('Status',        max_length = 1, default = 'A')
  client_id       = CharField('ID klienta',    max_length = 22, unique = True)
  card_id         = CharField('ID karty',      max_length = 22, unique = True)
  chip_id         = CharField('ID chipu',      max_length = 32, unique = True, null = True, blank = True)
  email           = CharField('Email',         max_length = 64,  null = True, blank = True)
  firstname       = CharField('Jméno',         max_length = 30,  null = True, blank = True)
  surname         = CharField('Příjmení',      max_length = 30,  null = True, blank = True)
  comment         = CharField('Komentář',      max_length = 128, null = True, blank = True)
  news            = BooleanField('Novinky',    default = False)
  daily_menu      = BooleanField('Denní menu', default = False)
  expiration_date = DateField('Platnost do',   null = True, blank = True)
  personal        = JSONField('Osobní',        default = dict)
  company         = JSONField('Firma',         default = dict)
  address         = JSONField('Adresa',        default = dict)
  address2        = JSONField('Adresa 2',      default = dict)
  created_at      = DateTimeField(auto_now_add = True)
  updated_at      = DateTimeField(auto_now = True)

  def __repr__(self):
    return "<%s %s '%s' '%s'>" % \
      (self.__class__.__name__, self.card_id, self.firstname, self.surname)

  def __str__(self):
    return "%s %s" % (self.firstname, self.surname)

class ClientInfo(Model):
  class Meta:
    db_table = 'client_info'

  client        = OneToOneField('Client', related_name = 'info', on_delete = CASCADE)
  typ           = CharField(max_length = 1, default = 'C')
  limit         = IntegerField(default = 0)
  limit_var     = IntegerField(default = 0)
  limit_den     = IntegerField(default = 0)
  sl_limit      = IntegerField(default = 0)
  sl_limit_mes  = IntegerField(default = 0)
  c_hlad        = SmallIntegerField(default = 1)
  pl_c_hlad     = SmallIntegerField(default = 1)
  typ_slevy     = SmallIntegerField(default = 1)
  pl_typ_slevy  = SmallIntegerField(default = 1)
  sl_koef       = DecimalField(max_digits = 3, decimal_places = 2, default = 0)
  pl_sl_koef    = DecimalField(max_digits = 3, decimal_places = 2, default = 0)
  omez_platba   = BooleanField(default = False)
  k_vyplate     = BooleanField(default = False)
  rod_cis       = CharField(max_length = 11, null = True, blank = True)
  id_vlastnik   = CharField(max_length = 2,  null = True, blank = True)
  pr_pozice     = CharField(max_length = 20, null = True, blank = True)
  st_autor      = CharField(max_length = 3,  null = True, blank = True)
  rezerva       = CharField(max_length = 30, null = True, blank = True)
  profil        = CharField(max_length = 5,  null = True, blank = True)
  max_den_odber = SmallIntegerField(null = True)
  akt_den_odber = SmallIntegerField(null = True)
  id_omezeni    = SmallIntegerField(null = True)
  posl_poh_nul  = SmallIntegerField(null = True)
  posl_poh_arc  = SmallIntegerField(null = True)
  id_firmy      = IntegerField(null = True)
  id_zuct       = IntegerField(null = True)
  sl_min_cast   = IntegerField(null = True)
  d_zm_status   = DateField(null = True)
  d_konec_kred  = DateField(null = True)
  d_posl_odber  = DateField(null = True)
  created_at    = DateTimeField(auto_now_add = True)
  updated_at    = DateTimeField(auto_now = True)

class Transaction(Model):
  class Meta:
    db_table            = 'transactions'
    verbose_name        = 'transakce'
    verbose_name_plural = 'transakce'

  id                  = BigAutoField(primary_key = True)
  client              = ForeignKey('Client', related_name = 'transactions', on_delete = PROTECT,
                                   verbose_name = 'klient')
  bill_header         = ForeignKey(BillHeader, related_name = 'transactions',
                                   on_delete = PROTECT, null = True, blank = True)
  kind                = CharField(max_length = 8)
  amount              = DecimalField(max_digits = 12, decimal_places = 2)
  balance             = DecimalField(max_digits = 12, decimal_places = 2)
  transacted_at       = DateTimeField(default = timezone.now)
  description         = CharField(max_length = 32,  null = True, blank = True)
  display_description = CharField(max_length = 128, null = True, blank = True)

  @classmethod
  def balance_for(cls, client, kind, at = None):
    last_transaction = cls.last_for(client, kind, at = at)

    return (last_transaction and last_transaction.balance) or 0

  @classmethod
  def balance_for_at(cls, client, kind, transacted_at):
    previous_transaction = cls.last_for_at(client, kind, transacted_at)

    return (previous_transaction and previous_transaction.balance) or 0

  @classmethod
  def last_for(cls, client, kind, at = None):
    filters = Q(client = client, kind = kind)

    if at: filters = filters & Q(transacted_at__lte = at)

    return cls.objects \
      .filter(filters) \
      .order_by('transacted_at') \
      .last()

  @classmethod
  def last_for_at(cls, client, kind, transacted_at):
    return cls.objects \
      .filter(client = client, kind = kind, transacted_at__lt = transacted_at) \
      .order_by('transacted_at') \
      .last()

  def __repr__(self):
    return ("<%s %s %s %s %s>" %
            (self.__class__.__name__, repr(self.client), self.kind, self.amount, self.balance))

  __str__ = __repr__

class KreditPermissions(Model):
  class Meta:
    managed = False

    permissions = [
       ('view_transaction_history', 'View transaction history')
    ]
