import sys, random
from decimal import Decimal
from datetime import date, timedelta
from faker    import Faker
from django.apps import apps
from django.db import transaction
from django.db.models import ProtectedError
from django.core.management.base import BaseCommand
from alto_django_utils.counter import Counter
from alto_django_food.models import Product
from alto_django_sk.models import BillHeader, BillItem, BillPayment
from alto_django_sk.serializers import BillItemSerializer
from alto_django_kredit.models import Client, Transaction
from alto_django_kredit.services import CreateTransaction

DEFAULT_TAX_RATE        = 21
DEFAULT_REGISTER_NUMBER = 123
NUMBER_FOR_EACH_KIND    = 10

fake = Faker()

def safe_encode(string):
  return string.encode('utf-8').decode(sys.getfilesystemencoding(), 'backslashreplace')

class Command(BaseCommand):
  help = 'Seed bills'

  @transaction.atomic
  def handle(self, *args, **options):
    random.seed()

    Transaction.objects.all().delete()
    BillItem.objects.all().delete()
    BillPayment.objects.all().delete()
    BillHeader.objects.all().delete()

    Product.objects.bulk_create(Product(name              = fake.sentence(5)[:64],
                                        price             = random.random() * 300,
                                        tax_rate          = DEFAULT_TAX_RATE,
                                        stock_card_number = random.randint(0, 10**9))
                                for i in range(20 - Product.objects.count()))

    counter  = Counter()
    products = Product.objects.all()

    for client in Client.objects.all().order_by('id'):
      print(f'creating bills for {safe_encode(str(client))}')

      for kind in ['KC', 'POI']:
        transaction = CreateTransaction(client = client, kind = kind, amount = 10000,
                                        description = counter.fetch_and_add())()

        for i in range(NUMBER_FOR_EACH_KIND):
          with_bill = random.choice([True, False])

          if with_bill:
            header, total = self.generate_bill(products, counter)
            bill_number   = header.c_uctu
          else:
            header        = None
            bill_number   = counter.fetch_and_add()
            total         = round(Decimal(random.random() * 1000), 2)

          transaction = CreateTransaction(client = client, kind = kind, amount = -total,
                                          description = bill_number, bill_header = header)()

  def generate_bill(self, products, counter):
    total  = Decimal(0)
    header = BillHeader.objects.create(c_kas  = DEFAULT_REGISTER_NUMBER,
                                       c_uctu = counter.fetch_and_add())

    for j in range(random.randint(1, 5)):
      product  = random.choice(products)
      quantity = random.randint(1, 5)
      price    = product.price * quantity
      wo_tax   = round(price * 100 / (100 + DEFAULT_TAX_RATE), 2)
      tax      = round(price * DEFAULT_TAX_RATE / (100 + DEFAULT_TAX_RATE), 2)

      data = {'header':     header,
              'product':    product,
              'mnozstvi':   quantity,
              'prachy':     price,
              'prachy_puv': wo_tax,
              'dan':        tax,
              'dan_sazba':  DEFAULT_TAX_RATE,
              'c_stredisk': DEFAULT_REGISTER_NUMBER}

      ser = BillItemSerializer(data = data)

      ser.is_valid(raise_exception = True)
      ser.save()

      total += price

    return header, total
