from pydash import _
from smtplib import SMTPException
from django.conf import settings
from django.forms import Form, CharField, EmailField
from django.core.mail import EmailMultiAlternatives
from django.template.loader import render_to_string
from rest_framework.views import APIView
from rest_framework.response import Response
from rest_framework.permissions import AllowAny
from alto_django_utils.exceptions import Misconfigured, MailSendingFailed

class SendSupportEmailForm(Form):
  reply_to    = EmailField()
  card_number = CharField()
  category    = CharField()
  subject     = CharField()
  message     = CharField()

class SendSupportEmail(APIView):
  permission_classes = [AllowAny]

  def post(self, request):
    data = request.data

    if request.user and getattr(request.user, 'client', None):
      data['reply_to']    = data.get('reply_to',    request.user.email)
      data['card_number'] = data.get('card_number', request.user.client.card_id)

    form = SendSupportEmailForm(data)

    if form.is_valid():
      self._send_email(form.cleaned_data)

      return Response()
    else:
      return Response(dict(form.errors), status = 400)

  def _send_email(self, data):
    from_email, support_email, support_cc, support_bcc = self._load_settings()
    reply_to, category, subject = _.at(data, 'reply_to', 'category', 'subject')

    email_subject = f'{category.capitalize()}: {subject}'
    template      = 'alto_django_mailing/support_email'
    plain         = render_to_string(template + '.txt',  data).strip()
    html          = render_to_string(template + '.html', data).strip()

    email = EmailMultiAlternatives(from_email = from_email,
                                   to         = [support_email],
                                   cc         = support_cc,
                                   bcc        = support_bcc,
                                   reply_to   = [reply_to],
                                   subject    = email_subject,
                                   body       = plain)

    email.attach_alternative(html, 'text/html')

    try:
      email.send()
    except SMTPException as error:
      raise MailSendingFailed(error)

  def _load_settings(self):
    try:
      from_email    = settings.DEFAULT_FROM_EMAIL
      support_email = settings.SUPPORT_EMAIL
      support_cc    = getattr(settings, 'SUPPORT_EMAIL_CC',  [])
      support_bcc   = getattr(settings, 'SUPPORT_EMAIL_BCC', [])

      return from_email, support_email, support_cc, support_bcc
    except AttributeError as error:
      raise Misconfigured(error)
