import pytest
from django.core import mail
from django.core.exceptions import FieldDoesNotExist
from django.urls import reverse
from django.conf import settings
from django.contrib.auth import get_user_model
from rest_framework.test import APITestCase
from alto_django_base_auth.tests.helpers import BaseAuthHelpers, auth_header

try:
  from alto_django_kredit.tests.factories.client import ClientFactory

  clients_supported = True
except ImportError:
  clients_supported = False

User = get_user_model()

try:
  User._meta.get_field('client')
except FieldDoesNotExist:
  clients_supported = False

class SendSupportEmailTests(APITestCase, BaseAuthHelpers):
  card_number = '123456'
  email       = 'test@test.cz'
  category    = 'Registration'
  subject     = 'Help!'
  message     = "It doesn't work"

  def setUp(self):
    User.objects.create_superuser('admin', 'admin@test.cz', 'admin')

    self.admin_token = self.get_token(username = 'admin', password = 'admin').data['token']

  def test(self):
    response = self.client.post(
      reverse('alto_django_mailing:send-support-email'), {
        'reply_to': self.email, 'card_number': self.card_number,
        'category': self.category, 'subject':  self.subject,
        'message':  self.message
      }
    )

    assert response.status_code == 200

    email = mail.outbox[0]

    assert email.from_email == settings.DEFAULT_FROM_EMAIL
    assert email.to         == [settings.SUPPORT_EMAIL]
    assert email.cc         == settings.SUPPORT_EMAIL_CC
    assert email.bcc        == settings.SUPPORT_EMAIL_BCC
    assert email.reply_to   == [self.email]
    assert email.subject    == f'{self.category}: {self.subject}'

    assert f'Číslo věrnostní karty: {self.card_number}' in email.body
    assert f'Kategorie: {self.category}'                in email.body
    assert f'Předmět: {self.subject}'                   in email.body
    assert f'Tělo zprávy:\n{self.message}'              in email.body

  def test_logged_out_user(self):
    response = self.client.post(
      reverse('alto_django_mailing:send-support-email'), {
        'category': self.category, 'subject': self.subject, 'message':  self.message
      }
    )

    assert response.status_code         == 400
    assert response.data['reply_to']    == ['Toto pole je vyžadováno.']
    assert response.data['card_number'] == ['Toto pole je vyžadováno.']

  @pytest.mark.skipif(not clients_supported, reason = 'needs ClientFactory and ClientUser')
  def test_logged_in_user(self):
    client = ClientFactory.create(card_id = self.card_number)
    user   = User.objects.create_user('test', self.email, 'test', client = client)
    token  = self.get_token(username = 'test', password = 'test').data['token']

    response = self.client.post(
      reverse('alto_django_mailing:send-support-email'), {
        'category': self.category, 'subject': self.subject, 'message':  self.message
      }, **auth_header(token)
    )

    assert response.status_code == 200

    email = mail.outbox[0]

    assert email.reply_to == [self.email]

    assert f'Číslo věrnostní karty: {self.card_number}' in email.body
