import re
from django.contrib.auth import get_user_model
from django.utils.translation import gettext as _
from rest_framework.views import APIView
from rest_framework.response import Response
from rest_framework.exceptions import ValidationError
from alto_django_utils.params import required_query_param
from alto_django_utils.permissions import AllowAny
from alto_django_kredit.models import Client
from alto_django_kredit.settings import settings

User = get_user_model()

email_regex = r"^[A-Za-z0-9\.\+_-]+@[A-Za-z0-9\._-]+\.[a-zA-Z]*$"

def mask_email(email):
  local, domain = email.split('@')
  stars = '*' * (len(local) - 2)

  return local[0] + stars + local[-1] + '@' + domain

class ClientByCard(APIView):
  permission_classes = [AllowAny]

  def get(self, request):
    card_number = required_query_param(request, 'card_number')

    try:
      clients = settings.client_finder.clients_for_card(card_number)
    except settings.client_finder.InvalidCardNumber:
      self.invalid_card_number(card_number)

    try:
      client = clients.get()

      return self.one_client_found(client)
    except Client.DoesNotExist:
      return self.no_clients_found()
    except Client.MultipleObjectsReturned:
      return self.many_clients_found(clients)

  def invalid_card_number(self, card_number):
    raise ValidationError({ 'card_number': [_('Not a valid card number')] })

  def one_client_found(self, client):
    try:
      client.user

      return Response({ 'found': 'one', 'registered': True })
    except User.DoesNotExist:
      has_valid_email = bool(client.email and re.fullmatch(email_regex, client.email))
      email           = mask_email(client.email) if has_valid_email else None

      return Response({ 'found': 'one', 'registered': False, 'email': email })

  def many_clients_found(self, clients):
    try:
      for client in clients: client.user
      return Response({ 'found': 'many', 'all_registered': True })
    except User.DoesNotExist:
      return Response({ 'found': 'many', 'all_registered': False })

  def no_clients_found(self):
    return Response({ 'found': 'none' })
