from pydash import _
from datetime import date
from django.http import HttpResponse
from rest_framework.views import APIView
from rest_framework.response import Response
from rest_framework.exceptions import NotFound, ValidationError
from rest_framework.decorators import api_view, permission_classes
from alto_django_utils.params import required_query_param, required_body_param
from alto_django_utils.permissions import AllowAny, IsAuthenticated, IsClientUser
from alto_django_kredit.models import ClientInfo
from alto_django_kredit.serializers import ClientSerializer
from alto_django_client_auth.serializers import UserSerializer
from alto_django_client_auth.services.profile import (
  EnsureProfileComplete, GetBarcode, GetClientBarcode,
)

def no_client_associated_with(username):
  return NotFound({ 'detail': f'Uživatel {username} není asociován s žádným klientem.' })

def no_client_info():
  return ValidationError({ 'detail': 'Klient nemá info záznam.' })

@api_view(['GET'])
@permission_classes([IsAuthenticated])
def get_client(request):
  if request.user.client:
    serializer  = ClientSerializer(request.user.client)

    try:
      client_type = request.user.client.info.typ
    except ClientInfo.DoesNotExist:
      client_type = None

    return Response({ 'client': serializer.data, 'type': client_type })
  else:
    raise no_client_associated_with(request.user.username)

class EditClient(APIView):
  permission_classes = [IsAuthenticated]

  def patch(self, request):
    if request.user.client:
      try:
        client_data = self._update_client(request)
      except ValidationError as error:
        raise ValidationError({ 'client': error.detail })

      try:
        user_data = self._update_user(request)
      except ValidationError as error:
        raise ValidationError({ 'user': error.detail })

      return Response({ 'client': client_data, 'user': user_data })
    else:
      raise no_client_associated_with(request.user.username)

  def _update_client(self, request):
    client = request.user.client

    old_birth_date = client.personal.get('birth_date')
    personal       = request.data.get('personal', client.personal)
    new_birth_date = isinstance(personal, dict) and personal.get('birth_date')

    if new_birth_date != old_birth_date:
      personal['birth_date_updated_at'] = date.today().isoformat()

    serializer = ClientSerializer(client, data = request.data, partial = True)

    serializer.is_valid(raise_exception = True)
    serializer.save()

    return serializer.data

  def _update_user(self, request):
    data = {}

    email, firstname, surname = _.at(request.data, 'email', 'firstname', 'surname')

    if firstname: data['first_name'] = firstname
    if surname:   data['last_name']  = surname

    if email:
      data['email'] = email

      if request.user.username == request.user.email:
        data['username'] = email

    serializer = UserSerializer(request.user, data = data, partial = True)

    serializer.is_valid(raise_exception = True)
    serializer.save()

    return serializer.data

@api_view(['PATCH'])
@permission_classes([IsAuthenticated])
def set_client_type(request):
  if request.user.client:
    try:
      client_type = required_body_param(request, 'type')
      info        = request.user.client.info
      info.typ    = client_type
      info.save()

      return Response(status = 204)
    except ClientInfo.DoesNotExist:
      raise no_client_info()
  else:
    raise no_client_associated_with(request.user.username)

@api_view(['GET'])
@permission_classes([IsAuthenticated])
def is_profile_complete(request):
  if not request.user.client:
    raise no_client_associated_with(request.user.username)

  try:
    EnsureProfileComplete(request.user.client)()

    return Response({ 'is_complete': True,  'detail': 'OK' })
  except (EnsureProfileComplete.MissingValue, EnsureProfileComplete.MissingField) as error:
    return Response({ 'is_complete': False, 'detail': str(error) })
  except EnsureProfileComplete.InvalidRequiredFieldsSetting as error:
    return Response({ 'detail': str(error) }, status = 412)

@api_view(['GET'])
@permission_classes([AllowAny])
def get_barcode(request):
  number = required_query_param(request, 'number')

  try:
    code, barcode = GetBarcode(number)()

    return HttpResponse(barcode, content_type = GetBarcode.MIME_TYPE)
  except (GetBarcode.BarcodesNotSupported, GetBarcode.BarcodesNotEnabled) as error:
    return Response({ 'detail': error.message, 'code': error.code }, status = 412)

@api_view(['GET'])
@permission_classes([IsClientUser])
def get_client_barcode(request):
  try:
    code, barcode = GetClientBarcode(request.user.client)()

    return HttpResponse(barcode, content_type = GetClientBarcode.MIME_TYPE)
  except (GetClientBarcode.BarcodesNotSupported, GetClientBarcode.BarcodesNotEnabled) as error:
    return Response({ 'detail': error.message, 'code': error.code }, status = 412)
