import re
from django.core import mail
from django.contrib.auth import get_user_model
from rest_framework.test import APITestCase
from alto_django_kredit.models import Client, Transaction
from alto_django_kredit.tests.factories.client import ClientFactory
from alto_django_kredit.settings import settings
from alto_django_kredit.tests.helpers import AmbiguousClientFinder
from alto_django_client_auth.tests.helpers import ClientAuthHelpers

User = get_user_model()

class ClientRegistrationWithBillVerificationTests(APITestCase, ClientAuthHelpers):
  def setUp(self):
    settings.client_finder = AmbiguousClientFinder()

  def tearDown(self):
    settings.reset()

  # general failures

  def test_invalid_card_number(self):
    response = self.register_client_with_bv(card_number = '1234567',
                                            bill_number = '1234', bill_date = '2000-01-01')

    assert response.status_code  == 400
    assert response.data['code'] == 'invalid_card_number'

  def test_client_not_found(self):
    response = self.register_client_with_bv(card_number = '123',
                                            bill_number = '1234', bill_date = '2000-01-01')

    assert response.status_code  == 400
    assert response.data['code'] == 'client_not_found'

  def test_bill_not_found(self):
    client = ClientFactory.create()

    response = self.register_client_with_bv(card_number = client.card_id, username = 'test',
                                            bill_number = '1234', bill_date = '2000-01-01')

    assert response.status_code == 400
    assert response.data['code'] == 'bill_not_found'

  def test_invalid_username(self):
    client = ClientFactory.create()

    transaction1 = Transaction.objects.create(client = client, kind = 'POI',
                                              amount = 10, balance = 10,
                                              transacted_at = '2000-01-01T12:00:00Z',
                                              description = '0001234')

    response = self.register_client_with_bv(card_number = client.card_id, username = '',
                                            bill_number = '1234', bill_date = '2000-01-01')

    assert response.status_code      == 400
    assert response.data['username'] == ['Toto pole nesmí být prázdné.']

  def test_invalid_password(self):
    client   = ClientFactory.create()

    transaction1 = Transaction.objects.create(client = client, kind = 'POI',
                                              amount = 10, balance = 10,
                                              transacted_at = '2000-01-01T12:00:00Z',
                                              description = '0001234')

    response = self.register_client_with_bv(card_number = client.card_id, username = 'test', password = 'test',
                                            bill_number = '1234', bill_date = '2000-01-01')

    self.assert_proper_password_validations(response)

  def test_multiple_bills_found(self):
    client = ClientFactory.create()

    transaction1 = Transaction.objects.create(client = client, kind = 'POI',
                                              amount = 10, balance = 10,
                                              transacted_at = '2000-01-01T12:00:00Z',
                                              description = '0001234')
    transaction2 = Transaction.objects.create(client = client, kind = 'POI',
                                              amount = 10, balance = 10,
                                              transacted_at = '2000-01-01T12:00:00Z',
                                              description = '0001234')

    response = self.register_client_with_bv(card_number = client.card_id, username = 'test',
                                            bill_number = '1234', bill_date = '2000-01-01')

    assert response.status_code  == 400
    assert response.data['code'] == 'multiple_bills_found'

  # single client for card

  def test_single_client__success(self):
    client = ClientFactory.create()
    transaction = Transaction.objects.create(client = client, kind = 'POI',
                                             amount = 10, balance = 10,
                                             transacted_at = '2000-01-01T12:00:00Z',
                                             description = '0001234')

    response = self.register_client_with_bv(card_number = client.card_id, username = 'test',
                                            bill_number = '1234', bill_date = '2000-01-01').data

    user = User.objects.first()

    assert user.username == 'test'

  def test_single_client__already_registered(self):
    client      = ClientFactory.create()
    user        = User.objects.create_user('test', 'test@test.cz', 'test', client = client)
    transaction = Transaction.objects.create(client = client, kind = 'POI',
                                             amount = 10, balance = 10,
                                             transacted_at = '2000-01-01T12:00:00Z',
                                             description = '0001234')

    response = self.register_client_with_bv(card_number = client.card_id, username = 'test',
                                            bill_number = '1234', bill_date = '2000-01-01')

    assert response.status_code  == 400
    assert response.data['code'] == 'client_already_registered'

  def test_single_client__bill_does_not_belong_to_client(self):
    client1 = ClientFactory.create(client_id = '123', card_id = '123')
    client2 = ClientFactory.create(client_id = '222', card_id = '222') # bill owner

    transaction = Transaction.objects.create(client = client2, kind = 'POI',
                                             amount = 10, balance = 10,
                                             transacted_at = '2000-01-01T12:00:00Z',
                                             description = '0001234')

    response = self.register_client_with_bv(card_number = client1.card_id, username = 'test',
                                            bill_number = '1234', bill_date = '2000-01-01')

    assert response.status_code  == 400
    assert response.data['code'] == 'bill_does_not_belong_to_client'

  # multiple clients for card

  def test_multiple_clients__success(self):
    client1  = ClientFactory.create(client_id = '112345', card_id = '112345')
    client2  = ClientFactory.create(client_id = '212345', card_id = '212345')

    transaction = Transaction.objects.create(client = client1, kind = 'POI',
                                             amount = 10, balance = 10,
                                             transacted_at = '2000-01-01T12:00:00Z',
                                             description = '0001234')

    response = self.register_client_with_bv(card_number = '12345', username = 'test',
                                            bill_number = '1234', bill_date = '2000-01-01').data

    user = User.objects.first()

    assert user.username == 'test'

  def test_multiple_clients__all_registered(self):
    client1  = ClientFactory.create(client_id = '112345', card_id = '112345')
    client2  = ClientFactory.create(client_id = '212345', card_id = '212345')

    user = User.objects.create_user('test', 'test@test.cz', 'test', client = client1)

    transaction = Transaction.objects.create(client = client1, kind = 'POI',
                                             amount = 10, balance = 10,
                                             transacted_at = '2000-01-01T12:00:00Z',
                                             description = '0001234')

    response = self.register_client_with_bv(card_number = '12345', username = 'test',
                                            bill_number = '1234', bill_date = '2000-01-01')

    assert response.status_code  == 400
    assert response.data['code'] == 'client_already_registered'

  def test_multiple_clients__bill_does_not_belong_to_any_client(self):
    client1  = ClientFactory.create(client_id = '112345', card_id = '112345')
    client2  = ClientFactory.create(client_id = '212345', card_id = '212345')
    client3  = ClientFactory.create(client_id = '333333', card_id = '333333') # bill owner

    transaction = Transaction.objects.create(client = client3, kind = 'POI',
                                             amount = 10, balance = 10,
                                             transacted_at = '2000-01-01T12:00:00Z',
                                             description = '0001234')

    response = self.register_client_with_bv(card_number = '12345', username = 'test',
                                            bill_number = '1234', bill_date = '2000-01-01')

    assert response.status_code  == 400
    assert response.data['code'] == 'bill_does_not_belong_to_client'
