import re
from django.core import mail
from django.contrib.auth import get_user_model
from rest_framework.test import APITestCase
from alto_django_settings.models import Setting
from alto_django_kredit.models import Client
from alto_django_kredit.settings import settings
from alto_django_kredit.tests.factories import ClientFactory
from alto_django_kredit.tests.helpers import AmbiguousClientFinder
from alto_django_client_auth.tests.helpers import ClientAuthHelpers

User = get_user_model()

class ClientRegistrationTests(APITestCase, ClientAuthHelpers):
  def setUp(self):
    settings.client_finder = AmbiguousClientFinder()

  def tearDown(self):
    settings.reset()

  # success

  def test_creates_inactive_unprivileged_user(self):
    client = ClientFactory.create()

    response = self.register_client(client.card_id, username = 'test')

    assert response.status_code == 200

    user = User.objects.first()

    assert user.username == 'test'
    assert not user.is_active
    assert not user.is_staff
    assert not user.is_superuser

  def test_reuses_client_info(self):
    client = ClientFactory.create()

    response = self.register_client(client.card_id)

    assert response.status_code == 200

    user = User.objects.first()

    assert user.email      == client.email
    assert user.first_name == client.firstname
    assert user.last_name  == client.surname

  def test_sends_activation_email(self):
    client   = ClientFactory.create()

    response = self.register_client(client.card_id)
    token    = self.extract_token_from_email(mail.outbox[0])

    assert re.fullmatch(r'[0-9a-z]{3}-[0-9a-f]{20}', token)

  def test_with_explicit_email(self):
    client   = ClientFactory.create(email = None)

    response = self.register_client(client.card_id, email = 'test@test.cz')

    client.refresh_from_db()

    user = User.objects.first()

    assert client.email == 'test@test.cz'
    assert user.email   == 'test@test.cz'

  def test_with_new_client_registration_allowed(self):
    Setting.objects.create(
      key = 'auth.registration.new_client_registration', value = {'allowed': True}
    )

    unused_card = ClientFactory.create(status = Client.Status.UNASSIGNED)

    response = self.register_new_client()

    assert response.status_code == 200

    unused_card.refresh_from_db()

    assert unused_card.status == Client.Status.ACTIVE

  # failure

  def test_invalid_card_number(self):
    response = self.register_client('1234567')

    assert response.status_code  == 400
    assert response.data['code'] == 'invalid_card_number'

  def test_client_not_found(self):
    response = self.register_client(123)

    assert response.status_code  == 400
    assert response.data['code'] == 'client_not_found'

  def test_new_client_registration_when_not_allowed(self):
    response = self.register_new_client()

    assert response.status_code  == 400
    assert response.data['card_number'][0].code == 'invalid'

  def test_new_client_registration_when_no_unused_card_exists(self):
    Setting.objects.create(
      key = 'auth.registration.new_client_registration', value = {'allowed': True}
    )

    response = self.register_new_client()

    assert response.status_code  == 400
    assert response.data['code'] == 'client_not_found'

  def test_multiple_clients_found(self):
    client1  = ClientFactory.create(client_id = '112345', card_id = '112345')
    client2  = ClientFactory.create(client_id = '212345', card_id = '212345')

    response = self.register_client('12345', username = 'test', password = 'test')

    assert response.status_code  == 400
    assert response.data['code'] == 'multiple_clients_found'

  def test_client_already_registered(self):
    client   = ClientFactory.create()
    user     = User.objects.create_user('test', 'test@test.cz', 'test', client = client)

    response = self.register_client(client.card_id)

    assert response.status_code  == 400
    assert response.data['code'] == 'client_already_registered'

  def test_email_already_registered(self):
    client   = ClientFactory.create(email = 'test@test.cz')
    user     = User.objects.create_user('test', 'test@test.cz', 'test')

    response = self.register_client(client.card_id)

    assert response.status_code  == 400
    assert response.data['code'] == 'email_already_registered'

  def test_invalid_username(self):
    client   = ClientFactory.create()

    response = self.register_client(client.card_id, username = '')

    assert response.status_code      == 400
    assert response.data['username'] == ['Toto pole nesmí být prázdné.']

  def test_invalid_password(self):
    client   = ClientFactory.create()

    response = self.register_client(client.card_id, username = 'test', password = 'test')

    self.assert_proper_password_validations(response)
