import flexmock
from django.contrib.auth import get_user_model
from rest_framework.test import APITestCase
from alto_django_settings.models import Setting
from alto_django_kredit.tests.factories.client import ClientFactory
from alto_django_client_auth.services import profile
from alto_django_client_auth.tests.helpers import ClientAuthHelpers

User = get_user_model()

class GetBarcodeTests(APITestCase, ClientAuthHelpers):
  def test_success(self):
    Setting.objects.create(key = 'auth.profile.show_barcode', value = True)

    response = self.get_barcode('12345')

    assert response.status_code == 200
    assert response.get('content-type') == 'image/png'

  def test_when_not_enabled(self):
    response = self.get_barcode('12345')

    assert response.status_code == 412
    assert response.data['code'] == 'not_enabled'

  def test_when_not_supported(self):
    flexmock(profile, barcodes_supported = False)

    response = self.get_barcode('12345')

    assert response.status_code == 412
    assert response.data['code'] == 'not_supported'

class GetClientBarcodeTests(APITestCase, ClientAuthHelpers):
  def setUp(self):
    self.clnt = ClientFactory.create()

    User.objects.create_user('user',  'user@test.cz',  'user',  client = self.clnt)
    User.objects.create_user('admin', 'admin@test.cz', 'admin', is_superuser = True)

  def test_success(self):
    Setting.objects.create(key = 'auth.profile.show_barcode', value = True)

    token    = self.get_token(username = 'user', password = 'user').data['token']
    response = self.get_client_barcode(token)

    assert response.status_code == 200
    assert response.get('content-type') == 'image/png'

  def test_when_not_authenticated(self):
    response = self.get_client_barcode(None)

    assert response.status_code         == 403
    assert response.get('content-type') == 'application/json'
    assert response.data['detail'].code == 'authentication_failed'

  def test_when_not_client(self):
    token    = self.get_token(username = 'admin', password = 'admin').data['token']
    response = self.get_client_barcode(token)

    assert response.status_code         == 403
    assert response.get('content-type') == 'application/json'
    assert response.data['detail'].code == 'permission_denied'

  def test_when_not_enabled(self):
    token    = self.get_token(username = 'user', password = 'user').data['token']
    response = self.get_client_barcode(token)

    assert response.status_code == 412
    assert response.data['code'] == 'not_enabled'

  def test_when_not_supported(self):
    flexmock(profile, barcodes_supported = False)

    token    = self.get_token(username = 'user', password = 'user').data['token']
    response = self.get_client_barcode(token)

    assert response.status_code == 412
    assert response.data['code'] == 'not_supported'
