from django.contrib.auth import get_user_model
from rest_framework.test import APITestCase
from alto_django_kredit.models import Client
from alto_django_client_auth.tests.helpers import ClientAuthHelpers

User = get_user_model()

class ClientUserDeletionTests(APITestCase, ClientAuthHelpers):
  def setUp(self):
    self.clnt = Client.objects.create(
      client_id = '123', card_id = '123',
      firstname = 'Testy', surname = 'Testson',
      email     = 'test@test.cz',
      personal  = { 'sex': 'M' },
      company   = { 'name': 'Alto' },
      address   = { 'city': 'Prague' },
      address2  = { 'city': 'Brno' }
    )
    self.clnt2 = Client.objects.create(client_id = '234', card_id = '234')

    User.objects.create_user('user',  'user@test.cz',  'user',  client = self.clnt)
    User.objects.create_user('admin', 'admin@test.cz', 'admin', is_superuser = True)

  def test_deletes_user(self):
    token = self.get_token(username = 'admin', password = 'admin').data['token']

    delete_response = self.delete_client_user(self.clnt.id, token)
    login_response  = self.get_token(username = 'user', password = 'user')

    assert delete_response.status_code == 204
    assert login_response.status_code  == 403

  def test_deletes_sensitive_client_data_if_asked(self):
    client = self.clnt
    token  = self.get_token(username = 'admin', password = 'admin').data['token']

    self.delete_client_user(client.id, token, clear_sensitive_data = True)

    client.refresh_from_db()

    assert not client.email
    assert client.personal == {}
    assert client.company  == {}
    assert client.address  == {}
    assert client.address2 == {}

  def test_does_not_delete_sensitive_data_by_default(self):
    client = self.clnt
    token  = self.get_token(username = 'admin', password = 'admin').data['token']

    self.delete_client_user(client.id, token)

    client.refresh_from_db()

    assert client.email
    assert client.personal['sex'] == 'M'

  def test_delete_by_nonadmin(self):
    token = self.get_token(username = 'user', password = 'user').data['token']

    delete_response = self.delete_client_user(self.clnt.id, token)
    login_response  = self.get_token(username = 'user', password = 'user')

    assert delete_response.status_code == 403
    assert login_response.status_code  == 200

  def test_client_not_registered(self):
    token  = self.get_token(username = 'admin', password = 'admin').data['token']

    response = self.delete_client_user(self.clnt2.id, token)

    assert response.status_code    == 404
    assert response.data['detail'] == f'Klient s id {self.clnt2.id} není registrován.'
