from django.conf import settings

try:
  import barcode
  from io        import BytesIO
  from math      import ceil
  from itertools import cycle

  barcodes_supported = True
except ImportError:
  barcodes_supported = False

from alto_django_settings.models import Setting

class EnsureProfileComplete:
  class InvalidRequiredFieldsSetting(Exception):
    def __init__(self):
      message = 'auth.profile.required_fields setting must be a list of dotted string paths'

      super().__init__(message)

  class MissingField(Exception):
    def __init__(self, field):
      super().__init__('Missing field ' + field)

  class MissingValue(Exception):
    def __init__(self, field):
      super().__init__('Missing value for ' + field)


  def __init__(self, client):
    self.client          = client
    self.required_fields = Setting.for_key('auth.profile.required_fields', [])

    if not self._is_valid_setting(self.required_fields):
      raise self.InvalidRequiredFieldsSetting()

  def _is_valid_setting(self, setting):
    return (isinstance(setting, list) and
            all(self._is_valid_field(field) for field in setting))

  def _is_valid_field(self, field):
    return isinstance(field, str) and \
      all(segment.isidentifier() for segment in field.split('.'))


  def __call__(self):
    for field in self.required_fields:
      value = self._get_field_value(self.client, field)

      if not (value or value == 0):
        raise self.MissingValue(field)

  def _get_field_value(self, client, field):
    segments = field.split('.')

    try:
      value = getattr(client, segments[0])
    except AttributeError:
      raise self.MissingField(field)

    for segment in segments[1:]:
      if isinstance(value, dict):
        try:
          value = value[segment]
        except KeyError:
          raise self.MissingField(field)
      else:
        raise self.MissingField(field)

    return value

class GetBarcode:
  class BarcodesNotSupported(RuntimeError):
    def __init__(self):
      self.message = "Barcodes aren't supported for this installation"
      self.code    = 'not_supported'

      super().__init__(self.message)

  class BarcodesNotEnabled(RuntimeError):
    def __init__(self):
      self.message = "Barcodes aren't enabled for this installation"
      self.code    = 'not_enabled'

      super().__init__(self.message)

  EAN_LENGTH = 13
  MIME_TYPE  = 'image/png'

  def __init__(self, number):
    if not barcodes_supported:                                  raise self.BarcodesNotSupported()
    if not Setting.for_key('auth.profile.show_barcode', False): raise self.BarcodesNotEnabled()

    self.number = number

  def __call__(self):
    code    = self.get_code(self.number)
    barcode = self.get_barcode(code)

    return code, barcode

  @classmethod
  def get_code(cls, number):
    code_without_parity = str(number).zfill(cls.EAN_LENGTH - 1)

    parity = cls.compute_parity(code_without_parity)

    return code_without_parity + parity

  @staticmethod
  def compute_parity(str_code_without_parity):
    digits    = map(int, str_code_without_parity)
    weights   = cycle([1, 3])
    products  = (digit * weight for digit, weight in zip(digits, weights))
    summation = sum(products)
    nearest10 = int(ceil(summation / 10.0)) * 10

    return str(nearest10 - summation)

  def get_barcode(self, code):
    output = BytesIO()

    barcode.generate('EAN13', code, barcode.writer.ImageWriter(), output)

    data = output.getvalue()

    output.close()

    return data

class GetClientBarcode(GetBarcode):
  class ClientHasNoChipID(RuntimeError):
    def __init__(self):
      super().__init__("Client has no chip ID")

  def __init__(self, client):
    if client.chip_id is None: raise self.ClientHasNoChipID()

    super().__init__(client.chip_id)
