from django.db               import IntegrityError
from rest_framework.response import Response
from dynamic_rest.viewsets   import DynamicModelViewSet, WithDynamicViewSetMixin
from .pagination             import CustomPagination

class CustomModelViewSet(DynamicModelViewSet):
  pagination_class = CustomPagination
  ordering_fields  = '__all__'
  ordering         = 'pk'

  def get_serializer(self, *args, **kwargs):
    if 'request_fields' not in kwargs:
      kwargs['request_fields'] = self.get_request_fields()
    if 'sideloading' not in kwargs:
      kwargs['sideloading'] = self.get_request_sideloading()
    if 'debug' not in kwargs:
      kwargs['debug'] = self.get_request_debug()
    if 'envelope' not in kwargs:
      kwargs['envelope'] = True
    # we dont want this, since it leads to update responses including deferred one-to-many relations
    # if self.is_update():
    #   kwargs['include_fields'] = '*'

    return super(WithDynamicViewSetMixin, self).get_serializer(*args, **kwargs)

  # don't check global permissions for detail routes
  def check_permissions(self, request):
    if self.action in ('retrieve', 'update', 'partial_update', 'destroy'): return

    method  = request.method.lower()
    handler = getattr(self, method, None)

    if handler and getattr(handler, 'detail', False): return

    return super().check_permissions(request)

  def handle_exception(self, exc):
    try:
      return super().handle_exception(exc)
    except IntegrityError as error:
      return Response({ 'detail': error.args[0] }, status = 400)

def viewset_for(model, serializer, **attrs):
  attrs['queryset']         = model.objects
  attrs['serializer_class'] = serializer

  return type(model.__name__ + 'ViewSet', (CustomModelViewSet,), attrs)
