import inspect
from contextlib import contextmanager
from django.utils.translation import gettext as _
from django.db import connection, transaction
from django.db.models import F, Model
from django.db.models.deletion import ProtectedError

@contextmanager
def exclusive_access_to(model_or_table):
  if isinstance(model_or_table, str):
    table = model_or_table
  elif inspect.isclass(model_or_table) and issubclass(model_or_table, Model):
    table = model_or_table._meta.db_table
  else:
    raise TypeError('exclusive_access_to must be called with either a table name or a model class')

  with transaction.atomic():
    with connection.cursor() as cursor:
      cursor.execute(f'lock table {table} in access exclusive mode')

    yield

def qs_to_sql(queryset):
  with connection.cursor() as cursor:
    sql, params = queryset.query.sql_with_params()
    cursor.execute('explain ' + sql, params)
    return cursor.query.decode()[8:] # remove explain

class FE(F):
  def __eq__(self, other):
    return self._combine(other, '=', False)

  def __ne__(self, other):
    return self._combine(other, '!=', False)

def PROTECT(collector, field, sub_objs, using):
  deleted_model     = field.remote_field.model
  referencing_model = sub_objs[0].__class__
  message           = _("Cannot delete some instances of model '%(deleted_model)s' because "
                        "they are referenced through a protected foreign key:"
                        "'%(referencing_model)s.%(referencing_field)s'")
  substitutions     = {
    'deleted_model':     deleted_model._meta.verbose_name.title(),
    'referencing_model': referencing_model._meta.verbose_name.title(),
    'referencing_field': field.verbose_name
  }

  raise ProtectedError(message % substitutions, sub_objs)
