import functools, operator
from pydash import py_
from django.db import DEFAULT_DB_ALIAS
from django.conf import settings
from django.apps import apps as global_apps
from django.db.models import Q
from django.contrib.auth.management import create_permissions
from django.contrib.contenttypes.management import create_contenttypes

def set_up_default_permissions(
  app_name, app_config, default_permissions = None, verbosity = 2, interactive = True,
  using = DEFAULT_DB_ALIAS, apps = global_apps, **kwargs
):
  if app_config.label != app_name: return
  if default_permissions is None: default_permissions = getattr(settings, 'DEFAULT_PERMISSIONS', {})
  if not default_permissions: return

  try:
    group_model      = apps.get_model('auth', 'Group')
    permission_model = apps.get_model('auth', 'Permission')
  except LookupError:
    return

  force_model_permissions_creation(
    app_config, default_permissions, verbosity = verbosity, interactive = interactive,
    using = using, apps = apps, **kwargs
  )

  for group_name, permission_names in default_permissions.items():
    ensure_group_permissions(
      group_name, permission_names, group_model, permission_model, using = using
    )

def force_model_permissions_creation(
  _app_config, default_permissions = {}, verbosity = 2, interactive = True,
  using = DEFAULT_DB_ALIAS, _apps = global_apps, **kwargs
):
  app_labels = py_.uniq(
    permission_name.split('.')[0] for permission_name in py_.flatten(default_permissions.values())
  )

  for app_label in app_labels:
    app_config = global_apps.get_app_config(app_label)

    create_contenttypes(
      app_config, verbosity = verbosity, interactive = interactive, using = using, **kwargs
    )
    create_permissions(
      app_config, verbosity = verbosity, interactive = interactive, using = using, **kwargs
    )

def ensure_group_permissions(group_name, permission_names, group_model, permission_model,
                             using = DEFAULT_DB_ALIAS):
  group       = group_model.objects.using(using).get(name = group_name)
  filters     = [
    Q(content_type__app_label = app_label, codename = codename)
    for (app_label, codename) in [pn.split('.') for pn in permission_names]
  ]
  permissions = (
    permission_model.objects.using(using)
    .filter(functools.reduce(operator.or_, filters))
    .values_list('id', flat = True)
  )

  group.permissions.add(*permissions)
