from rest_framework.response             import Response
from rest_framework.decorators           import api_view, permission_classes
from rest_framework.exceptions           import AuthenticationFailed
from rest_framework.permissions          import AllowAny
from alto_django_utils.params            import required_body_param
from alto_django_kredit.models           import Client
from alto_django_client_auth.views.login import GetClientToken, GetClientTokenPair
from alto_django_canteen.utils           import get_client_ip
from alto_django_canteen.models          import Terminal
from alto_django_canteen.serializers     import TerminalSerializer

@api_view(['GET'])
@permission_classes([AllowAny])
def is_registered_terminal(request):
  client_ip = get_client_ip(request)
  secret    = request.query_params.get('secret', '')
  terminal  = Terminal.objects.filter(ip = client_ip, secret = secret).first()

  return Response({ 'terminal': terminal and TerminalSerializer(terminal).data,
                    'is_registered': bool(terminal) })

class GetTerminalTokenMixin:
  def _authenticate(self, request):
    try:
      card_number = required_body_param(request, 'card_number')
      secret      = request.data.get('secret', '')
      client_ip   = get_client_ip(request)

      terminal    = Terminal.objects.get(ip = client_ip, secret = secret)
      client      = Client.objects.get(chip_id__iexact = card_number, user__isnull = False)
      user        = client.user

      if user.is_active:
        return { 'user': user, 'terminal': terminal }
      else:
        raise AuthenticationFailed({ 'detail': 'user not activated' })
    except Terminal.DoesNotExist:
      raise AuthenticationFailed({ 'detail': 'not a registered terminal' })
    except Client.DoesNotExist:
      raise AuthenticationFailed({ 'detail': 'bad credentials' })

  def _get_result(self, context):
    result = super()._get_result(context)

    result['terminal'] = TerminalSerializer(context['terminal']).data

    return result

class GetTerminalToken    (GetTerminalTokenMixin, GetClientToken    ): pass
class GetTerminalTokenPair(GetTerminalTokenMixin, GetClientTokenPair): pass

get_terminal_token      = GetTerminalToken.as_view()
get_terminal_token_pair = GetTerminalTokenPair.as_view()
