from django.forms import Form, IntegerField, DateField, ModelChoiceField, ModelMultipleChoiceField
from django.shortcuts import get_object_or_404
from rest_framework.response import Response
from rest_framework.decorators import api_view, permission_classes
from alto_django_utils.params import required_body_param
from alto_django_utils.permissions import IsClientUser, IsAdminOrStaff
from alto_django_kredit.models import Client
from alto_django_canteen.models import Canteen, Meal, Course, Order
from alto_django_canteen.serializers import OrderSerializer
from alto_django_canteen.services.preorder_meals import PreorderMeals

class OrderMealForm(Form):
  date    = DateField()
  canteen = ModelChoiceField(queryset = Canteen.objects.all())
  course  = ModelChoiceField(queryset = Course.objects.all())
  number  = IntegerField()

@api_view(['POST'])
@permission_classes([IsClientUser])
def order_meal(request):
  client = request.user.client
  form   = OrderMealForm(data = request.data)

  if form.is_valid():
    data = form.cleaned_data
    meal = get_object_or_404(Meal,
                             date    = data['date'],
                             canteen = data['canteen'],
                             course  = data['course'],
                             number  = data['number'])

    try:
      order = Order.objects.get(client        = client,
                                meal__date    = data['date'],
                                meal__canteen = data['canteen'],
                                meal__course  = data['course'])

      order.meal = meal
      order.save()
    except Order.DoesNotExist:
      order = Order.objects.create(client = client, meal = meal)

    return Response({ 'order': OrderSerializer(order).data })
  else:
    return Response(dict(form.errors), status = 400)


class CancelMealForm(Form):
  date    = DateField()
  canteen = ModelChoiceField(queryset = Canteen.objects.all())
  course  = ModelChoiceField(queryset = Course.objects.all())

@api_view(['DELETE'])
@permission_classes([IsClientUser])
def cancel_meal(request):
  client = request.user.client
  form   = CancelMealForm(data = request.data)

  if form.is_valid():
    data  = form.cleaned_data
    order = get_object_or_404(Order, client = client,
                              meal__date    = data['date'],
                              meal__canteen = data['canteen'],
                              meal__course  = data['course'])

    order.delete()

    return Response(status = 204)
  else:
    return Response(dict(form.errors), status = 400)


class PreorderMealsForm(Form):
  client         = ModelChoiceField(queryset = Client.objects.all())
  canteen        = ModelChoiceField(queryset = Canteen.objects.all())
  from_date      = DateField()
  to_date        = DateField()
  from_course    = ModelChoiceField(        queryset = Course.objects.all())
  to_course      = ModelChoiceField(        queryset = Course.objects.all())
  only_courses   = ModelMultipleChoiceField(queryset = Course.objects.all(), required = False)
  except_courses = ModelMultipleChoiceField(queryset = Course.objects.all(), required = False)

@api_view(['POST'])
@permission_classes([IsAdminOrStaff])
def preorder_meals(request):
  form = PreorderMealsForm(request.data)

  if form.is_valid():
    try:
      orders = PreorderMeals(**form.cleaned_data)()

      return Response({ 'orders': OrderSerializer(orders, many = True).data })

    except (PreorderMeals.InvalidDateSelection,
            PreorderMeals.InvalidCourseSelection) as error:
      return Response({ 'detail': str(error) }, status = 400)
  else:
    return Response(dict(form.errors), status = 400)
