import re
from django.contrib.auth import get_user_model
from rest_framework.test import APITestCase
from alto_django_client_auth.tests.helpers import ClientAuthHelpers
from alto_django_canteen.tests.helpers import CanteenHelpers

User       = get_user_model()
uuid_regex = '[a-f0-9]{8}-?[a-f0-9]{4}-?4[a-f0-9]{3}-?[89ab][a-f0-9]{3}-?[a-f0-9]{12}'

class TerminalRegistrationTests(APITestCase, ClientAuthHelpers, CanteenHelpers):
  def setUp(self):
    self.admin       = User.objects.create_superuser('admin', 'admin@test.cz', 'admin')
    self.admin_token = self.get_token(username = 'admin',  password = 'admin' ).data['token']

  def test_create_with_defaults(self):
    response = self.register_terminal(name = 'test', token = self.admin_token)

    assert response.status_code              == 201
    assert response.data['terminal']['name'] == 'test'
    assert response.data['terminal']['ip']   == '127.0.0.1'
    assert re.fullmatch(uuid_regex, response.data['terminal']['secret'])

  def test_create_all_params_specified(self):
    response = self.register_terminal(name = 'test', ip = '1.2.3.4', secret = 'my secret',
                                      token = self.admin_token)

    assert response.status_code                == 201
    assert response.data['terminal']['name']   == 'test'
    assert response.data['terminal']['ip']     == '1.2.3.4'
    assert response.data['terminal']['secret'] == 'my secret'

  def test_update_updates_name(self):
    terminal_id = self.register_terminal(name = 'test', ip = '1.2.3.4', secret = 'my secret',
                                         token = self.admin_token).data['terminal']['id']

    response = self.update_terminal(terminal_id, name = 'updated', token = self.admin_token)

    assert response.status_code              == 200
    assert response.data['terminal']['name'] == 'updated'

  def test_update_keeps_ip_if_not_specified(self):
    terminal_id = self.register_terminal(name = 'test', ip = '1.2.3.4', secret = 'my secret',
                                         token = self.admin_token).data['terminal']['id']

    response = self.update_terminal(terminal_id, name = 'test', token = self.admin_token)

    assert response.status_code            == 200
    assert response.data['terminal']['ip'] == '1.2.3.4'

  def test_update_sets_ip_if_specified(self):
    terminal_id = self.register_terminal(name = 'test', ip = '1.2.3.4', secret = 'my secret',
                                         token = self.admin_token).data['terminal']['id']

    response = self.update_terminal(terminal_id, name = 'test', ip = '2.3.4.5',
                                    token = self.admin_token)

    assert response.status_code            == 200
    assert response.data['terminal']['ip'] == '2.3.4.5'

  def test_update_keeps_secret_if_not_specified(self):
    terminal_id = self.register_terminal(name = 'test', ip = '1.2.3.4', secret = 'my secret',
                                         token = self.admin_token).data['terminal']['id']

    response = self.update_terminal(terminal_id, name = 'test', token = self.admin_token)

    assert response.status_code                == 200
    assert response.data['terminal']['secret'] == 'my secret'

  def test_update_sets_secret_if_specified(self):
    terminal_id = self.register_terminal(name = 'test', ip = '1.2.3.4', secret = 'my secret',
                                         token = self.admin_token).data['terminal']['id']

    response = self.update_terminal(terminal_id, name = 'test', secret = 'updated',
                                    token = self.admin_token)

    assert response.status_code                == 200
    assert response.data['terminal']['secret'] == 'updated'

  def test_update_generates_secret_if_not_specified_and_not_yet_set(self):
    terminal_id = self.register_terminal(name = 'test', ip = '1.2.3.4', secret = '',
                                         token = self.admin_token).data['terminal']['id']

    response = self.update_terminal(terminal_id, name = 'test', token = self.admin_token)

    assert response.status_code == 200
    assert re.fullmatch(uuid_regex, response.data['terminal']['secret'])
