from faker import Faker
from datetime import date
from django.contrib.auth import get_user_model
from rest_framework.test import APITestCase
from alto_django_client_auth.tests.helpers import ClientAuthHelpers
from alto_django_kredit.models import Client
from alto_django_canteen.models import Canteen, Meal, Course, Order
from alto_django_canteen.tests.helpers import CanteenHelpers
from alto_django_canteen.tests.factories import MealFactory

fake = Faker()
User = get_user_model()

class MealOrderingTests(APITestCase, ClientAuthHelpers, CanteenHelpers):
  def setUp(self):
    self.client_    = Client.objects.create(client_id = '123', card_id = '123',
                                            email = 'client@test.cz')
    self.user       = User.objects.create_user('test', 'test@test.cz', 'test',
                                               client = self.client_)
    self.user_token = self.get_token(username = 'test',  password = 'test' ).data['token']

    self.canteen    = Canteen.objects.create(name = 'canteen')
    self.breakfast  = Course.objects.get(name = 'breakfast')
    self.lunch      = Course.objects.get(name = 'lunch')

    MealFactory.create_many(5, canteen = self.canteen, course = self.breakfast)

  def test_order_meal_with_no_existing_order(self):
    response = self.order_meal(date.today(), self.canteen, self.breakfast, 1, self.user_token)

    order = Order.objects.get()

    assert response.status_code == 200

    assert order.client      == self.client_
    assert order.meal.date   == date.today()
    assert order.meal.course == self.breakfast
    assert order.meal.number == 1

  def test_order_meal_with_existing_order(self):
    Order.objects.create(client = self.client_, meal = Meal.objects.last())

    response = self.order_meal(date.today(), self.canteen, self.breakfast, 1, self.user_token)

    order = Order.objects.get()

    assert response.status_code == 200

    assert order.client      == self.client_
    assert order.meal.date   == date.today()
    assert order.meal.course == self.breakfast
    assert order.meal.number == 1

  def test_order_meal_with_nonexistent_meal(self):
    response = self.order_meal(date.today(), self.canteen, self.lunch, 1, self.user_token)

    assert response.status_code == 404

  def test_cancel_meal_with_existing_order(self):
    Order.objects.create(client = self.client_, meal = Meal.objects.first())

    response = self.cancel_meal(date.today(), self.canteen, self.breakfast, self.user_token)

    assert response.status_code == 204
    assert not Order.objects.first()

  def test_cancel_meal_with_no_existing_order(self):
    response = self.cancel_meal(date.today(), self.canteen, self.breakfast, self.user_token)

    assert response.status_code == 404
