from pydash import _
from datetime import date, timedelta
from django.contrib.auth import get_user_model
from rest_framework.test import APITestCase
from alto_django_client_auth.tests.helpers import ClientAuthHelpers
from alto_django_kredit.models import Client
from alto_django_canteen.models import Meal, Order
from alto_django_canteen.tests.helpers import CanteenHelpers
from alto_django_canteen.tests.factories import CanteenFactory, CourseFactory, MealFactory

today = date.today()
User  = get_user_model()

class LoadMenuTests(APITestCase, ClientAuthHelpers, CanteenHelpers):
  def setUp(self):
    self.client1    = Client.objects.create(client_id = '123', card_id = '123',
                                            email = 'client1@test.cz')
    self.client2    = Client.objects.create(client_id = '234', card_id = '234',
                                            email = 'client2@test.cz')
    self.user       = User.objects.create_user('test', 'test@test.cz', 'test',
                                               client = self.client1)
    self.user_token = self.get_token(username = 'test',  password = 'test' ).data['token']

    dates   = [today + timedelta(days = i) for i in range(7)]
    courses = [CourseFactory.breakfast, CourseFactory.lunch_soup, CourseFactory.lunch_main,
               CourseFactory.lunch_side, CourseFactory.dinner]
    meals   = [MealFactory.build_many(3, date = date, course = course,
                                      canteen = CanteenFactory.default)
               for date in dates for course in courses]

    Meal.objects.bulk_create(_.flatten(meals))

  def test(self):
    response = self.load_menu()

    canteens, course_groups, courses, meals = \
      _.at(response.data, 'canteens', 'course_groups', 'courses', 'meals')

    assert len(canteens) == 1
    assert _.find(canteens, {'id': CanteenFactory.default.id})

    assert len(course_groups) == 3
    assert _.find(course_groups, {'name': CourseFactory.lunch_main.group.name})

    assert len(courses) == 5
    assert _.find(courses, {'name': CourseFactory.lunch_main.name})

    assert len(meals) == 7 * 5 * 3 # 7 days, 5 courses, 3 meals for each
    assert _.find(meals, {'canteen': CanteenFactory.default.id,     'date': today.isoformat(),
                          'course':  CourseFactory.lunch_main.name, 'number': 1})

  def test_with_explicit_bounds(self):
    from_date = today + timedelta(days = 1)
    to_date   = today + timedelta(days = 3)
    day_after = today + timedelta(days = 4)

    response = self.load_menu(from_date = from_date, to_date = to_date)
    meals    = response.data['meals']

    assert len(meals) == 3 * 5 * 3 # 3 days, 5 courses, 3 meals for each

    assert not _.find(meals, {'date': today.isoformat()})
    assert     _.find(meals, {'date': from_date.isoformat()})
    assert     _.find(meals, {'date': to_date.isoformat()})
    assert not _.find(meals, {'date': day_after.isoformat()})

  def test_with_orders(self):
    meal = Meal.objects.first()

    Order.objects.create(client = self.client1, meal = meal)
    Order.objects.create(client = self.client2, meal = meal)

    response = self.load_menu(with_orders = True, token = self.user_token)

    assert     _.find(response.data['orders'], {'client': self.client1.id, 'meal': meal.id})
    assert not _.find(response.data['orders'], {'client': self.client2.id, 'meal': meal.id})
