import pytest
from django.urls                         import reverse
from alto_django_base_auth.tests.helpers import auth_header
from alto_django_settings.models         import Setting

@pytest.fixture
def incomplete_terminal_user_credentials(card_number_factory):
  return card_number_factory()

@pytest.fixture
def incomplete_terminal_user(user_factory, client_factory, incomplete_terminal_user_credentials):
  client = client_factory(chip_id = incomplete_terminal_user_credentials['card_number'],
                          firstname = '', surname = '', email = '')

  return user_factory(username = 'incomplete_terminal_user', client = client,
                      first_name = 'Incomplete Terminal', last_name = 'User',
                      email = 'incomplete_terminal_user@test.cz')

pytestmark = pytest.mark.django_db()

@pytest.mark.parametrize(
  '  url_name,                 use_jwt, token_key',
  (('get-terminal-token',      False,   'token'),
   ('v1-get-terminal-token',   False,   'token'),
   ('get-terminal-token-pair', True,    'access_token')))
def test_terminal_authentication(api_client, url_name, use_jwt, token_key, terminal_user, terminal,
                                 terminal_user_credentials):
  credentials = { **terminal_user_credentials, 'secret': terminal.secret }

  get_response = api_client.post(reverse(f'alto_django_canteen:{url_name}'), credentials)

  assert get_response.status_code == 200

  test_response = api_client.get(
    reverse('alto_django_base_auth:test-token'),
    **auth_header(token = get_response.data[token_key], use_jwt = use_jwt))

  assert test_response.status_code == 200

@pytest.mark.parametrize(
  'url_name',
  ('get-terminal-token', 'v1-get-terminal-token', 'get-terminal-token-pair'))
class TestCommonTerminalAuthCases:
  def test_terminal_user_presence(self, api_client, url_name, terminal_user, terminal,
                                  terminal_user_credentials):
    credentials = { **terminal_user_credentials, 'secret': terminal.secret }

    response = api_client.post(reverse(f'alto_django_canteen:{url_name}'), credentials)

    assert response.status_code               == 200
    assert response.data['user']['username']  == terminal_user.username
    assert response.data['client']['card_id'] == terminal_user.client.card_id

  def test_with_complete_profile(self, api_client, url_name, terminal_user, terminal,
                                 terminal_user_credentials):
    credentials = { **terminal_user_credentials, 'secret': terminal.secret }

    response = api_client.post(reverse(f'alto_django_canteen:{url_name}'), credentials)

    assert response.status_code              == 200
    assert response.data['profile_complete'] is True

  def test_with_incomplete_profile(self, api_client, url_name, incomplete_terminal_user,
                                   terminal, incomplete_terminal_user_credentials):
    credentials = { **incomplete_terminal_user_credentials, 'secret': terminal.secret }

    Setting.objects.create(key   = 'auth.profile.required_fields',
                           value = ['firstname', 'surname', 'email'])

    response = api_client.post(reverse(f'alto_django_canteen:{url_name}'), credentials)

    assert response.status_code              == 200
    assert response.data['profile_complete'] is False

  # TODO add test case for existing client
  def test_terminal_not_registered(self, api_client, url_name):
    credentials = { 'card_number': '123', 'secret': None }

    response = api_client.post(reverse(f'alto_django_canteen:{url_name}'), credentials)

    assert response.status_code    == 403
    assert response.data['detail'] == 'not a registered terminal'

  def test_bad_secret(self, api_client, url_name, terminal_user,
                      terminal, terminal_user_credentials):
    credentials = { **terminal_user_credentials, 'secret': 'bad' + terminal.secret }

    response = api_client.post(reverse(f'alto_django_canteen:{url_name}'), credentials)

    assert response.status_code    == 403
    assert response.data['detail'] == 'not a registered terminal'

  def test_client_not_found(self, api_client, url_name, terminal, terminal_user_credentials):
    credentials = { 'card_number': 'bad' + terminal_user_credentials['card_number'],
                    'secret':      terminal.secret }

    response = api_client.post(reverse(f'alto_django_canteen:{url_name}'), credentials)

    assert response.status_code    == 403
    assert response.data['detail'] == 'bad credentials'

  def test_client_not_registered(self, api_client, url_name, terminal_client, terminal,
                                 terminal_user_credentials):
    credentials = { **terminal_user_credentials, 'secret': terminal.secret }

    response = api_client.post(reverse(f'alto_django_canteen:{url_name}'), credentials)

    assert response.status_code    == 403
    assert response.data['detail'] == 'bad credentials'
