from datetime import date
from django.contrib.auth import get_user_model
from rest_framework.test import APITestCase
from alto_django_client_auth.tests.helpers import ClientAuthHelpers
from alto_django_kredit.models import Client
from alto_django_canteen.models import Canteen, Meal, CourseGroup, Order
from alto_django_canteen.serializers import OrderSerializer
from alto_django_canteen.tests.helpers import CanteenHelpers
from alto_django_canteen.tests.factories import MealFactory, CourseFactory

User = get_user_model()

class CourseGroupOrderingTests(APITestCase, ClientAuthHelpers, CanteenHelpers):
  def setUp(self):
    self.client_    = Client.objects.create(client_id = '123', card_id = '123',
                                            email = 'client@test.cz')
    self.user       = User.objects.create_user('test', 'test@test.cz', 'test',
                                               client = self.client_)
    self.user_token = self.get_token(username = 'test',  password = 'test' ).data['token']

    self.canteen = Canteen.objects.create(name = 'canteen')
    self.lunch   = CourseGroup.objects.get(name = 'lunch')
    self.soup    = CourseFactory.create(name = 'soup',   group = self.lunch)
    self.main    = CourseFactory.create(name = 'main',   group = self.lunch)
    self.side    = CourseFactory.create(name = 'side',   group = self.lunch, is_additional = True)
    self.desert  = CourseFactory.create(name = 'desert', group = self.lunch)

    MealFactory.create_many(3, date = date.today(), canteen = self.canteen, course = self.soup)
    MealFactory.create_many(3, date = date.today(), canteen = self.canteen, course = self.main)
    MealFactory.create_many(3, date = date.today(), canteen = self.canteen, course = self.side)
    MealFactory.create_many(3, date = date.today(), canteen = self.canteen, course = self.desert)

  def test_creates_orders(self):
    orders = [
      {'course': 'soup',   'number': 1},
      {'course': 'main',   'number': 2},
      {'course': 'desert', 'number': 3},
    ]

    response = self.order_course_group(date.today(), self.canteen, self.lunch, orders,
                                       self.user_token)

    assert response.status_code  == 200
    assert Order.objects.count() == 3

    for order in orders:
      meal = Meal.objects.get(date = date.today(), canteen = self.canteen,
                              course = order['course'], number = order['number'])

      assert Order.objects.filter(client = self.client_, meal = meal).exists()

  def test_returns_orders(self):
    orders = [
      {'course': 'soup',   'number': 1},
      {'course': 'main',   'number': 2},
      {'course': 'desert', 'number': 3},
    ]

    response = self.order_course_group(date.today(), self.canteen, self.lunch, orders,
                                       self.user_token)

    assert response.status_code  == 200
    assert response.data         == {
      'orders': OrderSerializer(Order.objects.all(), many = True).data
    }

  def test_updates_existing_orders(self):
    meal  = Meal.objects.get(date = date.today(), canteen = self.canteen,
                             course = 'main', number = 1)
    order = Order.objects.create(client = self.client_, meal = meal)

    orders = [
      {'course': 'soup',   'number': 1},
      {'course': 'main',   'number': 2},
      {'course': 'desert', 'number': 3},
    ]

    response = self.order_course_group(date.today(), self.canteen, self.lunch, orders,
                                       self.user_token)

    order.refresh_from_db()

    assert response.status_code  == 200
    assert Order.objects.count() == 3
    assert order.meal.number     == 2

  def test_cancels_orders_for_null_numbers(self):
    orders = [
      {'course': 'soup',   'number': 1},
      {'course': 'main',   'number': 2},
      {'course': 'desert', 'number': 3},
    ]

    response = self.order_course_group(date.today(), self.canteen, self.lunch, orders,
                                       self.user_token)

    assert response.status_code  == 200
    assert Order.objects.count() == 3

    orders = [
      {'course': 'soup',   'number': 1},
      {'course': 'main',   'number': None},
      {'course': 'desert', 'number': 3},
    ]

    response = self.order_course_group(date.today(), self.canteen, self.lunch, orders,
                                       self.user_token)

    assert response.status_code  == 200
    assert Order.objects.count() == 2

  def test_nonexistent_meal_number(self):
    orders = [
      {'course': 'soup',   'number': 1},
      {'course': 'main',   'number': 5},
      {'course': 'desert', 'number': 3},
    ]

    response = self.order_course_group(date.today(), self.canteen, self.lunch, orders,
                                       self.user_token)

    assert response.status_code    == 400
    assert response.data['orders'] == [{'course': 'main', 'number': 'Jídlo s číslem 5 neexistuje'}]

  def test_orders_exist_in_other_canteen(self):
    other_canteen = Canteen.objects.create(name = 'other_canteen')
    meal = MealFactory.create(date = date.today(), canteen = other_canteen,
                              course = self.main, number = 1)
    Order.objects.create(client = self.client_, meal = meal)

    response = self.order_course_group(date.today(), self.canteen, self.lunch,
                                       [{'course': 'main', 'number': 1}],
                                       self.user_token)

    assert response.status_code     == 400
    assert response.data['canteen'] == ['Existují objednávky v jiné jídelně']

  def test_duplicate_course_orders(self):
    orders = [
      {'course': 'soup', 'number': 1},
      {'course': 'soup', 'number': 2},
    ]

    response = self.order_course_group(date.today(), self.canteen, self.lunch, orders,
                                       self.user_token)

    assert response.status_code    == 400
    assert response.data['orders'] == ["Duplicitní objednávka chodů ['soup']"]

  def test_additional_course_not_ordered_but_has_default(self):
    meal_with_ac = MealFactory.create(date = date.today(), canteen = self.canteen,
                                      course = self.main, number = 4,
                                      additional_course = self.side,
                                      additional_course_default_number = 1)

    response = self.order_course_group(date.today(), self.canteen, self.lunch,
                                       [{'course': 'main', 'number': 4}],
                                       self.user_token)

    assert response.status_code  == 200
    assert Order.objects.count() == 2
    assert Order.objects.filter(client = self.client_, meal = meal_with_ac).exists()

  def test_additional_course_not_ordered_and_default_doesnt_exist(self):
    meal_with_ac = MealFactory.create(date = date.today(), canteen = self.canteen,
                                      course = self.main, number = 4,
                                      additional_course = self.side,
                                      additional_course_default_number = 5)

    response = self.order_course_group(date.today(), self.canteen, self.lunch,
                                       [{'course': 'main', 'number': 4}],
                                       self.user_token)

    assert response.status_code    == 400
    assert response.data['orders'] == [{'course': 'main',
                                        'code': 'default_additional_meal_does_not_exist',
                                        'detail': 'Standardní doplňkové jídlo neexistuje'}]

  def test_additional_course_not_ordered_no_default(self):
    meal_with_ac = MealFactory.create(date = date.today(), canteen = self.canteen,
                                      course = self.main, number = 4,
                                      additional_course = self.side)

    response = self.order_course_group(date.today(), self.canteen, self.lunch,
                                       [{'course': 'main', 'number': 4}],
                                       self.user_token)

    assert response.status_code    == 400
    assert response.data['orders'] == [{'course': 'main', 'code': 'additional_meal_not_ordered',
                                        'detail': 'Doplňkové jídlo nebylo objednáno'}]

  def test_additional_course_ordered_but_no_main(self):
    response = self.order_course_group(date.today(), self.canteen, self.lunch,
                                       [{'course': 'side', 'number': 1}],
                                       self.user_token)

    assert response.status_code    == 400
    assert response.data['orders'] == [{'course': 'side', 'code': 'main_meal_not_ordered',
                                        'detail': 'Hlavní jídlo nebylo objednáno'}]
