from pydash import _
from datetime import timedelta
from alto_django_canteen.models import Course, Meal, Order

# TODO: move most of the course selecting logic to CourseList

class PreorderMeals:
  class InvalidDateSelection(Exception):
    def __init__(self, message):
      # TODO: translate this
      super().__init__('Invalid date selection: ' + message)

  class InvalidCourseSelection(Exception):
    def __init__(self, message):
      # TODO: translate this
      super().__init__('Invalid course selection: ' + message)

  def __init__(self, *, client, canteen, from_date, to_date, from_course, to_course,
               only_courses, except_courses):
    self.client      = client
    self.canteen     = canteen
    self.from_date   = from_date
    self.from_course = from_course
    self.to_date     = to_date
    self.to_course   = to_course
    # intersect + union might be more intuitive, but qs returned by those can't be filtered further
    self.courses     = (only_courses or Course.objects.all()).exclude(pk__in = except_courses)

  def __call__(self):
    number_of_days   = (self.to_date - self.from_date).days + 1

    if number_of_days < 1:
      # TODO: translate this
      raise self.InvalidDateSelection('from_date is later than to_date')
    elif number_of_days == 1:
      return self.one_day()
    else:
      return self.more_days(number_of_days)

  def one_day(self):
    if self.from_course.order_number > self.to_course.order_number:
      # TODO: translate this
      raise self.InvalidCourseSelection('from_course is later than to_course')

    courses = self.courses.filter(order_number__range = (self.from_course.order_number,
                                                         self.to_course.order_number))

    return Order.objects.bulk_create(self.default_order_for(self.from_date, course)
                                     for course in courses)

  def more_days(self, number_of_days):
    first_day_courses = self.courses.filter(order_number__gte = self.from_course.order_number)
    last_day_courses  = self.courses.filter(order_number__lte = self.to_course.order_number)

    inner_days = [self.from_date + timedelta(days = day + 1) for day in range(number_of_days - 2)]

    first_day_orders  = [self.default_order_for(self.from_date, course)
                         for course in first_day_courses]
    last_day_orders   = [self.default_order_for(self.to_date, course)
                         for course in last_day_courses]
    inner_days_orders = [self.default_order_for(date, course)
                         for date   in inner_days
                         for course in self.courses]

    return Order.objects.bulk_create(first_day_orders + inner_days_orders + last_day_orders)

  def default_order_for(self, date, course):
    meal = self.first_meal_for_course(date, course)

    return meal and Order(client = self.client, meal = meal)

  # ideally these would be preloaded in one query
  def first_meal_for_course(self, date, course):
    return Meal.objects.filter(canteen = self.canteen, date = date, course = course) \
      .order_by('number').first()
