from pydash import _
from django.db import transaction, IntegrityError
from alto_django_canteen.models import Meal, Order

class OrderCourseGroupService:
  class Error(Exception): pass

  class ErrorWithMeal(Error):
    def __init__(self, meal):
      self.meal = meal

  class OrdersInOtherCanteenExist(Error): pass
  class OrderIsAlreadyBeingCreated(ErrorWithMeal): pass
  class MainMealNotOrdered(ErrorWithMeal): pass
  class AdditionalMealNotOrdered(ErrorWithMeal): pass
  class DefaultAdditionalMealDoesNotExist(ErrorWithMeal): pass

  def __init__(self, client, *, date, canteen, group, orders):
    self.client  = client
    self.canteen = canteen
    self.date    = date
    self.group   = group
    self.orders  = orders

  @transaction.atomic
  def __call__(self):
    orders_in_other_canteens = Order.objects \
      .filter(client = self.client, meal__date = self.date, meal__course__group = self.group) \
      .exclude(meal__canteen = self.canteen)

    if orders_in_other_canteens.exists(): raise self.OrdersInOtherCanteenExist

    orders = _.compact(self.get_order(order['course'], order['meal']) for order in self.orders)
    default_ac_orders = self.check_ac_orders(orders)

    orders.extend(default_ac_orders)

    return orders

  def get_order(self, course, meal):
    order = Order.objects.filter(client        = self.client,
                                 meal__date    = self.date,
                                 meal__canteen = self.canteen,
                                 meal__course  = course).first()

    if meal:
      if order:
        order.meal = meal
        order.save()
      else:
        try: # race condition check
          order = Order.objects.create(client = self.client, meal = meal)
        except IntegrityError:
          raise self.OrderIsAlreadyBeingCreated(meal)
    else:
      if order:
        order.delete()
        order = None

    return order

  def check_ac_orders(self, orders):
    main_orders = _.filter(orders, lambda order: order.meal.additional_course)
    ac_orders   = _.filter(orders, lambda order: order.meal.course.is_additional)

    for ac_order in ac_orders: self.check_main_meal_ordered(main_orders, ac_order)

    return _.compact([self.check_additional_meal_ordered(main_order, ac_orders)
                      for main_order in main_orders])

  def check_main_meal_ordered(self, main_orders, ac_order):
    # test cancellation orders
    additional_course = ac_order.meal.course

    if not _.find(main_orders,
                  lambda main_order: main_order.meal.additional_course == additional_course):
      raise self.MainMealNotOrdered(ac_order.meal)

  def check_additional_meal_ordered(self, main_order, ac_orders):
    # test cancellation orders
    main_meal         = main_order.meal
    additional_course = main_meal.additional_course
    default_number    = main_meal.additional_course_default_number

    if not _.find(ac_orders, lambda ac_order: ac_order.meal.course == additional_course):
      if default_number:
        try:
          meal = Meal.objects.get(date = self.date, canteen = self.canteen,
                                  course = additional_course, number = default_number)
        except Meal.DoesNotExist:
          raise self.DefaultAdditionalMealDoesNotExist(main_meal)

        return Order.objects.create(client = self.client, meal = meal)
      else:
        raise self.AdditionalMealNotOrdered(main_meal)
