import os, pytest
import http.server
from threading import Thread
from unittest import TestCase
from datetime import timedelta
from flexmock import flexmock, flexmock_teardown
from alto_utils.loaders import LocalLoader, RemoteLoader, LoaderError, FileNotFound
from alto_utils.file_caching import FileCache, CachedFileReader


project_root = os.path.join(os.path.dirname(__file__), '../..')

class StaticFileHandler(http.server.SimpleHTTPRequestHandler):
  directory = project_root

  def log_message(*args, **kwargs): pass


class TestFileCaching(TestCase):
  @classmethod
  def setUpClass(cls):
    cls.server = http.server.HTTPServer(('localhost', 8888), StaticFileHandler)

  @classmethod
  def tearDownClass(cls):
    cls.server = None

  def setUp(self):
    cache_dir = os.path.join(project_root, 'tmp/file_cache')

    os.makedirs(cache_dir, exist_ok = True)

    local_loader     = LocalLoader(cache_dir)
    remote_loader    = RemoteLoader(base_url = 'http://localhost:8888')
    self.reader      = CachedFileReader(remote_loader, local_loader, reload_after_minutes = 60)
    self.static_file = 'setup.py'

  def tearDown(self):
    CachedFileReader.cache = FileCache()
    flexmock_teardown()


  def test_successful_read_caching(self):
    flexmock(self.reader.remote_loader).should_call('read_with_mtime').once()
    flexmock(self.reader.remote_loader).should_call('read_if_modified_after').never()

    self._handle_next_request()

    self.reader(self.static_file)
    self.reader(self.static_file)

  def test_successful_reload_when_expired(self):
    flexmock(self.reader.remote_loader).should_call('read_with_mtime').once()

    self._handle_next_request()
    self.reader(self.static_file)

    self._make_cached_record_outdated(self.static_file)

    flexmock(self.reader.remote_loader).should_call('read_if_modified_after').once()

    self._handle_next_request()
    self.reader(self.static_file)

  def test_failed_read_caching(self):
    flexmock(self.reader.remote_loader).should_call('read_with_mtime').once()
    flexmock(self.reader.remote_loader).should_call('read_if_modified_after').never()

    with pytest.raises(FileNotFound):
      self._handle_next_request()
      self.reader('non-existent.yml')

    with pytest.raises(FileNotFound):
      self.reader('non-existent.yml')

  def test_use_cached_on_network_error_even_if_outdated(self):
    self._handle_next_request()

    config1 = self.reader(self.static_file) # load and cache

    self._make_cached_record_outdated(self.static_file)

    # break connection to server
    flexmock(self.reader.remote_loader) \
      .should_receive('read_if_modified_after').and_raise(LoaderError)

    config2 = self.reader(self.static_file) # load from cache

    assert config1 == config2

  def test_use_file_from_disk_on_remote_loader_error(self):
    self._handle_next_request()

    config1 = self.reader(self.static_file) # load from server and store on disk

    CachedFileReader.cache = FileCache() # empty cache (new process)

    # break connection to server
    flexmock(self.reader.remote_loader).should_receive('read_with_mtime').and_raise(LoaderError)

    # verify reading from disk
    flexmock(self.reader.local_loader).should_call('read_with_mtime').once()

    config2 = self.reader(self.static_file) # load from disk

    assert config1 == config2

  def test_delete_file_on_disk_on_server_404(self):
    self._handle_next_request()
    self.reader(self.static_file) # load from server and store on disk

    CachedFileReader.cache = FileCache() # empty cache (new process)

    # remove from server
    flexmock(self.reader.remote_loader).should_receive('read_with_mtime').and_raise(FileNotFound)

    # remove from disk on server 404
    with pytest.raises(FileNotFound):
      self.reader(self.static_file)

    CachedFileReader.cache = FileCache() # empty cache (new process)

    # break connection to server
    flexmock(self.reader.remote_loader).should_receive('read_with_mtime') \
      .and_raise(LoaderError('broken for testing'))

    # try loading from disk and fail
    with pytest.raises(LoaderError, match = r'.*broken for testing.*'):
      self.reader(self.static_file) # load from disk


  def _handle_next_request(self):
    Thread(target = self.server.handle_request).start()

  @staticmethod
  def _make_cached_record_outdated(name):
    (cached, last_check, stored_mtime) = CachedFileReader.cache.get(name)
    CachedFileReader.cache.set(name, cached, last_check - timedelta(days = 365), stored_mtime)
