import re
from datetime import datetime
from .datetime import parse_to_aware

MONEY_REGEX = r'^-?(0|[1-9][0-9]*)\.[0-9]{2}$'

def string_to_bool(string):
  lower = string.lower()
  if   lower in ['true', '1']:  return True
  elif lower in ['false', '0']: return False
  else: raise ValueError("can't convert %s to bool" % string)

class ParsingError(Exception): pass

class MissingParam(ParsingError):
  def __init__(self, attr):
    super().__init__('missing attribute: %s' % attr)

class BadParamType(ParsingError):
  def __init__(self, attr, value, expected_type):
    super().__init__('bad %s attribute type: %s, expected %s' %
                     (attr, value, expected_type.__name__))

class BadParamFormat(ParsingError):
  def __init__(self, attr, value, expected_format = None):
    message = "bad %s attribute format: %s" % (attr, value)

    if expected_format is not None:
      message += " expected '%s'" % expected_format

    super().__init__(message)


class Param:
  def __init__(self, name, value):
    self.name  = name
    self.value = value

  @property
  def present(self):
    return self.value is not None

  @property
  def required(self):
    if self.value is None:
      raise MissingParam(self.name)

    return self

  def type(self, expected_type):
    if self.present and not isinstance(self.value, expected_type):
      raise BadParamType(self.name, self.value, expected_type)

    return self

  @property
  def string(self):
    return self.type(str) if self.present else self

  def format(self, expected_format):
    if self.present and not re.match(expected_format, self.value):
      raise BadParamFormat(self.name, self.value, expected_format)

    return self

  @property
  def money(self):
    return self.string and self.format(MONEY_REGEX)

  @property
  def timestamp(self):
    if self.present and self.string:
      timestamp = parse_to_aware(self.value)

      if not isinstance(timestamp, datetime):
        raise BadParamFormat(self.name, self.value, 'ISO8601 timestamp')

      return self.__class__(self.name, timestamp.astimezone())

    return self

  @property
  def boolean(self):
    if self.present:
      if isinstance(self.value, bool):
        return self
      elif isinstance(self.value, str):
        try:
          return self.__class__(self.name, string_to_bool(self.value))
        except ValueError:
          raise BadParamType(self.name, self.value, bool)
      else:
        raise BadParamType(self.name, self.value, bool)

class Params:
  def __init__(self, params):
    self.params = params

  def get(self, attr, default = None):
    return Param(attr, self.params.get(attr, default))
