import os
from time import sleep
from datetime import datetime
from contextlib import contextmanager
from . import http
from .printing import Printable
from .datetime import local_to_aware, parse_to_aware
from .exceptions import error_translation

@contextmanager
def lock_for(path):
  lock_path = path + '.lock'

  while True:
    try:
      open(lock_path, 'x').close() # acquire lock
      yield
      os.remove(lock_path) # release lock
      break
    except FileExistsError: # failed to acquire lock
      sleep(0.050)
    except Exception as exception: # exception in yield
      os.remove(lock_path) # release lock
      raise exception

class LoaderError(Exception): pass
class FileNotFound(Exception): pass

class LocalLoader(Printable):
  def __init__(self, root_dir):
    self.root_dir = root_dir

  def read_with_mtime(self, name, mode = 'r'):
    return self._read_with_mtime(self.full_path(name), mode)

  def write(self, name, contents, mode = 'w'):
    path = self.full_path(name)

    with lock_for(path):
      with open(path, mode) as file:
        file.write(contents)

  def delete(self, name):
    path = self.full_path(name)

    with lock_for(path):
      if os.path.exists(path):
        os.remove(path)

  def full_path(self, name):
    return os.path.join(self.root_dir, name)

  @error_translation(FileNotFoundError, FileNotFound)
  def _read_with_mtime(self, path, mode = 'r'):
    with lock_for(path):
      with open(path, mode) as file:
        (contents, mtime) = (file.read(), self._mtime(path))
      return (contents, mtime)

  @error_translation(FileNotFoundError, FileNotFound)
  def _mtime(self, path):
    return local_to_aware(datetime.fromtimestamp(os.path.getmtime(path)))

class RemoteLoader(Printable):
  def __init__(self, base_url, *, username = None, password = None,
               headers = {}, proxy = None, verify = True, logger = None, timeout = 10):
    self.client  = http.HTTPClient(base_url, username, password, proxy, verify, logger)
    self.headers = headers
    self.timeout = timeout

  def read_with_mtime(self, name, mode = None):
    response = self._get(name)

    return (response.body, self._mtime(response))

  def read_if_modified_after(self, name, timestamp, mode = None):
    response = self._get(name)
    mtime    = self._mtime(response)

    if mtime > timestamp:
      return (response.body, mtime)
    else:
      return (None, None)

  def _mtime(self, response):
    return parse_to_aware(response.headers.get('Last-Modified'))

  def _get(self, name):
    try:
      response = self.client.get('/' + name, headers = self.headers, timeout = self.timeout)

      if response.status == 200:
        return response
      if response.status == 404:
        raise FileNotFound('%s not found on server' % name)
      else:
        raise LoaderError('got non-200 response when loading %s: %s' % (name, response.status))

    except http.HTTPError as e:
      raise LoaderError("can't access key server: %s" % e)
