from urllib.parse import urlparse
from attr import attrs, attrib
from requests import request
from .exceptions import error_translation

@attrs(kw_only = True)
class HTTPResponse:
  status  = attrib(default = 200)
  headers = attrib(factory = dict)
  body    = attrib(default = None)
  data    = attrib(factory = None)

  @status.validator
  def validate(self, attribute, value):
    if not (isinstance(value, int) and 200 <= value < 600):
      raise ValueError('{value} is not a valid status value'.format(value = repr(value)))

  @classmethod
  def from_requests_response(cls, rresponse):
    is_json = rresponse.headers.get('content-type') == 'application/json'

    return cls(status  = rresponse.status_code,
               headers = rresponse.headers,
               body    = rresponse.content,
               data    = rresponse.json() if is_json else None)

class HTTPError(Exception):
  def __init__(self, cause):
    super().__init__(str(cause))

class HTTPClient:
  def __init__(self, base_url, username = '', password = '',
               proxy = None, verify = True, logger = None):

    if not urlparse(base_url).scheme: base_url = 'http://' + base_url

    self.base_url = base_url
    self.auth     = None
    self.proxy    = proxy
    self.verify   = verify
    self.logger   = logger

    if username:
      self.auth = (username, password)

  @error_translation(OSError, HTTPError)
  def request(self, method, path, body = '', headers = {}, timeout = 10, json = None,
              params = None):
    options = { 'headers': headers, 'timeout': timeout, 'verify': self.verify, 'json': json,
                'params': params }

    if body:
      options['data'] = body

    if self.auth:
      options['auth'] = self.auth

    if self.proxy:
      options['proxies'] = { 'http': self.proxy, 'https': self.proxy }

    if self.logger:
      self.logger.make_request(self.base_url, path, self.proxy)

    url = self.base_url + path

    return HTTPResponse.from_requests_response(request(method, url, **options))

  def get(self, path, headers = {}, timeout = 10, params = None):
    return self.request('GET', path, headers = headers, timeout = timeout,
                        params = params)

  def post(self, path, body = '', headers = {}, timeout = 10, json = None, params = None):
    return self.request('POST', path, body, headers, timeout = timeout, json = json,
                        params = params)

  def put(self, path, body = '', headers = {}, timeout = 10, json = None, params = None):
    return self.request('PUT', path, body, headers, timeout = timeout, json = json,
                        params = params)

  def patch(self, path, body = '', headers = {}, timeout = 10, json = None, params = None):
    return self.request('PATCH', path, body, headers, timeout = timeout, json = json,
                        params = params)

  def delete(self, path, body = '', headers = {}, timeout = 10, json = None, params = None):
    return self.request('DELETE', path, body, headers, timeout = timeout, json = json,
                        params = params)
